"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseQueryFilterToKQL = exports.parsePoliciesToKQL = exports.parsePoliciesAndFilterToKql = exports.getPolicyQuery = exports.getAddedFieldsCounts = exports.computeHasDuplicateFields = void 0;
var _fp = require("lodash/fp");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const parseQueryFilterToKQL = (filter, fields) => {
  if (!filter) return '';
  const kuery = fields.map(field => `exception-list-agnostic.attributes.${field}:(*${filter.trim().replace(/([\)\(\<\>\}\{\"\:\\])/gm, '\\$&') // Escape problematic chars
  .replace(/\s/gm, '*')}*)`).join(' OR ');
  return `(${kuery})`;
};
exports.parseQueryFilterToKQL = parseQueryFilterToKQL;
const getPolicyQuery = policyId => {
  if (policyId === 'global') return 'exception-list-agnostic.attributes.tags:"policy:all"';
  if (policyId === 'unassigned') return '(not exception-list-agnostic.attributes.tags:policy\\:*)';
  return `exception-list-agnostic.attributes.tags:"policy:${policyId}"`;
};
exports.getPolicyQuery = getPolicyQuery;
const parsePoliciesToKQL = (includedPolicies, excludedPolicies = []) => {
  if ((0, _fp.isEmpty)(includedPolicies) && (0, _fp.isEmpty)(excludedPolicies)) return '';
  const includedPoliciesKuery = includedPolicies.map(getPolicyQuery).join(' OR ');
  const excludedPoliciesKuery = excludedPolicies.map(policyId => `not ${getPolicyQuery(policyId)}`).join(' AND ');
  const kuery = [];
  if (includedPoliciesKuery) kuery.push(includedPoliciesKuery);
  if (excludedPoliciesKuery) kuery.push(excludedPoliciesKuery);
  return `(${kuery.join(' AND ')})`;
};

/**
 * Takes a list of policies (string[]) and an existing kuery
 * (string) and returns an unified KQL with and AND
 * The policy list can also contain "unassigned" and "global".
 * @param policies string[] a list of policies ids.
 * @param excludedPolicies string[] a list of policies ids to exclude.
 * @param kuery string an existing KQL.
 */
exports.parsePoliciesToKQL = parsePoliciesToKQL;
const parsePoliciesAndFilterToKql = ({
  policies = [],
  excludedPolicies = [],
  kuery
}) => {
  if ((policies === null || policies === void 0 ? void 0 : policies.length) === 0 && (excludedPolicies === null || excludedPolicies === void 0 ? void 0 : excludedPolicies.length) === 0) {
    return kuery ? kuery : undefined;
  }
  const policiesKQL = parsePoliciesToKQL(policies, excludedPolicies);
  return `(${policiesKQL})${kuery ? ` AND (${kuery})` : ''}`;
};

/**
 * Counts the number of conditions added for a new Event filter or Trusted app
 * @param formFields
 * @returns number of fields
 */
exports.parsePoliciesAndFilterToKql = parsePoliciesAndFilterToKql;
const getAddedFieldsCounts = formFields => formFields.reduce((allFields, field) => {
  if (field in allFields) {
    allFields[field]++;
  } else {
    allFields[field] = 1;
  }
  return allFields;
}, {});

/**
 * Checks if conditions in Event filters or Trusted Apps forms have duplicate fields
 * @param formFields
 * @returns boolean
 */
exports.getAddedFieldsCounts = getAddedFieldsCounts;
const computeHasDuplicateFields = formFieldsMap => Object.values(formFieldsMap).some(e => e > 1);
exports.computeHasDuplicateFields = computeHasDuplicateFields;