"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useFetchEntityDetailsHighlights = void 0;
var _react = require("react");
var _elasticAssistant = require("@kbn/elastic-assistant");
var _i18n = require("@kbn/i18n");
var _use_global_time = require("../../../../common/containers/use_global_time");
var _use_app_toasts = require("../../../../common/hooks/use_app_toasts");
var _api = require("../../../api/api");
var _helpers = require("../utils/helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useFetchEntityDetailsHighlights = ({
  connectorId,
  anonymizationFields,
  entityType,
  entityIdentifier
}) => {
  const {
    fetchEntityDetailsHighlights
  } = (0, _api.useEntityAnalyticsRoutes)();
  const {
    addError
  } = (0, _use_app_toasts.useAppToasts)();
  const {
    from,
    to
  } = (0, _use_global_time.useGlobalTime)();
  const [assistantResult, setAssistantResult] = (0, _react.useState)(null);
  const {
    abortStream,
    sendMessage,
    isLoading: isChatLoading
  } = (0, _elasticAssistant.useChatComplete)({
    connectorId
  });
  const fetchEntityHighlights = (0, _react.useCallback)(async () => {
    const errorTitle = _i18n.i18n.translate('xpack.securitySolution.flyout.entityDetails.highlights.fetch.errorTitle', {
      defaultMessage: `Failed to run LLM`
    });
    const {
      summary,
      replacements,
      prompt
    } = await fetchEntityDetailsHighlights({
      entityType,
      entityIdentifier,
      anonymizationFields,
      from: new Date(from).getTime(),
      to: new Date(to).getTime(),
      connectorId
    }).catch(error => {
      addError(error, {
        title: errorTitle
      });
      return {
        summary: null,
        replacements: null,
        prompt: null
      };
    });
    if (!summary || !replacements || !prompt) {
      return;
    }
    const summaryFormatted = JSON.stringify(summary);
    const rawResponse = await sendMessage({
      message: `${prompt}.      
        Context:
            EntityType: ${entityType},
            EntityIdentifier: ${(0, _helpers.getAnonymizedEntityIdentifier)(entityIdentifier, replacements)},
          ${summaryFormatted}
        `,
      replacements,
      query: {
        content_references_disabled: true
      }
    });
    if (rawResponse.isError) {
      addError(new Error(rawResponse.response), {
        title: errorTitle
      });
      return;
    }
    setAssistantResult({
      formattedEntitySummary: summaryFormatted,
      aiResponse: rawResponse.response,
      replacements
    });
  }, [fetchEntityDetailsHighlights, entityType, entityIdentifier, anonymizationFields, from, to, connectorId, sendMessage, addError]);
  return {
    fetchEntityHighlights,
    isChatLoading,
    abortStream,
    result: assistantResult
  };
};
exports.useFetchEntityDetailsHighlights = useFetchEntityDetailsHighlights;