"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ScheduleRequestHandler = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _configSchema = require("@kbn/config-schema");
var _lodash = require("lodash");
var _server = require("@kbn/task-manager-plugin/server");
var _server2 = require("@kbn/core/server");
var _reportingServer = require("@kbn/reporting-server");
var _latest = require("../../../saved_objects/scheduled_report/schemas/latest");
var _saved_objects = require("../../../saved_objects");
var _request_handler = require("./request_handler");
var _lib = require("./lib");
var _audit_events = require("../../../services/audit_events/audit_events");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Using the limit specified in the cloud email service limits
// https://www.elastic.co/docs/explore-analyze/alerts-cases/watcher/enable-watcher#cloud-email-service-limits
const MAX_ALLOWED_EMAILS = 30;
const validation = {
  params: _configSchema.schema.object({
    exportType: _configSchema.schema.string({
      minLength: 2
    })
  }),
  body: _configSchema.schema.object({
    schedule: _server.scheduleRruleSchemaV3,
    notification: _configSchema.schema.maybe(_latest.rawNotificationSchema),
    jobParams: _configSchema.schema.string()
  }),
  query: _configSchema.schema.nullable(_configSchema.schema.object({}))
};

/**
 * Handles the common parts of requests to generate a report
 * Serves report job handling in the context of the request to generate the report
 */
class ScheduleRequestHandler extends _request_handler.RequestHandler {
  async checkLicenseAndTimezone(exportTypeId, browserTimezone) {
    const {
      reporting,
      res
    } = this.opts;
    const licenseInfo = await reporting.getLicenseInfo();
    const licenseResults = licenseInfo.scheduledReports;
    if (!licenseResults.enableLinks) {
      return res.forbidden({
        body: licenseResults.message
      });
    }
    return super.checkLicenseAndTimezone(exportTypeId, browserTimezone);
  }
  static getValidation() {
    return validation;
  }
  getSchedule() {
    let rruleDef = null;
    const req = this.opts.req;
    const res = this.opts.res;
    const {
      schedule
    } = req.body;
    const {
      rrule
    } = schedule !== null && schedule !== void 0 ? schedule : {};
    rruleDef = rrule;
    if ((0, _lodash.isEmpty)(rruleDef)) {
      throw res.customError({
        statusCode: 400,
        body: 'A schedule is required to create a scheduled report.'
      });
    }
    if (rruleDef.dtstart && !(0, _moment.default)(rruleDef.dtstart).isValid()) {
      throw res.customError({
        statusCode: 400,
        body: `Invalid startedAt date: ${rruleDef.dtstart}`
      });
    }
    return schedule;
  }
  getNotification() {
    const {
      reporting,
      req,
      res
    } = this.opts;
    const {
      notification
    } = req.body;
    if ((0, _lodash.isEmpty)(notification) || (0, _lodash.isEmpty)(notification.email)) {
      return undefined;
    }
    const allEmails = new Set([...(notification.email.to || []), ...(notification.email.bcc || []), ...(notification.email.cc || [])]);
    if (allEmails.size === 0) {
      return undefined;
    }
    if (allEmails.size > MAX_ALLOWED_EMAILS) {
      throw res.customError({
        statusCode: 400,
        body: `Maximum number of recipients exceeded: cannot specify more than ${MAX_ALLOWED_EMAILS} recipients.`
      });
    }
    const invalidEmails = reporting.validateNotificationEmails([...allEmails]);
    if (invalidEmails) {
      throw res.customError({
        statusCode: 400,
        body: `Invalid email address(es): ${invalidEmails}`
      });
    }
    return notification;
  }
  async enqueueJob(params) {
    var _jobParams$layout;
    const {
      id,
      exportTypeId,
      jobParams,
      schedule,
      notification
    } = params;
    const {
      reporting,
      logger,
      req,
      user
    } = this.opts;
    const soClient = await reporting.getScopedSoClient(req);
    const auditLogger = await reporting.getAuditLogger(req);
    const {
      version,
      job,
      jobType,
      name
    } = await this.createJob(exportTypeId, jobParams);
    const reportId = id || _server2.SavedObjectsUtils.generateId();
    auditLogger.log((0, _audit_events.scheduledReportAuditEvent)({
      action: _audit_events.ScheduledReportAuditAction.SCHEDULE,
      savedObject: {
        type: _saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE,
        id: reportId,
        name: job.title
      },
      outcome: 'unknown'
    }));
    const payload = {
      ...job,
      title: job.title,
      objectType: jobParams.objectType,
      browserTimezone: jobParams.browserTimezone,
      version,
      spaceId: reporting.getSpaceId(req, logger)
    };

    // TODO - extract saved object references before persisting

    const attributes = {
      createdAt: _moment.default.utc().toISOString(),
      // we've already checked that user exists in handleRequest
      // this fallback is just to satisfy the type
      createdBy: user ? user.username : 'unknown',
      enabled: true,
      jobType,
      meta: {
        // telemetry fields
        isDeprecated: job.isDeprecated,
        layout: (_jobParams$layout = jobParams.layout) === null || _jobParams$layout === void 0 ? void 0 : _jobParams$layout.id,
        objectType: jobParams.objectType
      },
      migrationVersion: version,
      ...(notification ? {
        notification
      } : {}),
      title: job.title,
      payload: JSON.stringify((0, _lodash.omit)(payload, 'forceNow')),
      schedule: schedule
    };

    // Create a scheduled_report saved object
    const report = await soClient.create(_saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE, attributes, {
      id: reportId
    });
    logger.debug(`Successfully created scheduled report: ${report.id}`);

    // Schedule the report with Task Manager
    const task = await reporting.scheduleRecurringTask(req, (0, _lib.transformRawScheduledReportToTaskParams)(report));
    logger.info(`Scheduled "${name}" reporting task. Task ID: task:${task.id}. Report ID: ${report.id}`);
    return (0, _lib.transformRawScheduledReportToReport)(report);
  }
  async handleRequest(params) {
    const {
      exportTypeId,
      jobParams
    } = params;
    const {
      reporting,
      req,
      res
    } = this.opts;
    const checkErrorResponse = await this.checkLicenseAndTimezone(exportTypeId, jobParams.browserTimezone);
    if (checkErrorResponse) {
      return checkErrorResponse;
    }

    // check that security requirements are met
    const reportingHealth = await reporting.getHealthInfo();
    if (!reportingHealth.hasPermanentEncryptionKey) {
      return res.forbidden({
        body: `Permanent encryption key must be set for scheduled reporting`
      });
    }
    if (!reportingHealth.isSufficientlySecure) {
      return res.forbidden({
        body: `Security and API keys must be enabled for scheduled reporting`
      });
    }
    // check that username exists
    if (!this.opts.user || !this.opts.user.username) {
      return res.forbidden({
        body: `User must be authenticated to schedule a report`
      });
    }
    const auditLogger = await reporting.getAuditLogger(req);
    let report;
    const id = _server2.SavedObjectsUtils.generateId();
    try {
      report = await this.enqueueJob({
        ...params,
        id
      });
      const eventTracker = reporting.getEventTracker(report.id, exportTypeId, jobParams.objectType);
      eventTracker === null || eventTracker === void 0 ? void 0 : eventTracker.createReport({
        isDeprecated: Boolean(report.payload.isDeprecated),
        isPublicApi: false,
        scheduleType: _reportingServer.ScheduleType.SCHEDULED
      });
      return res.ok({
        headers: {
          'content-type': 'application/json'
        },
        body: {
          job: report
        }
      });
    } catch (err) {
      var _report;
      auditLogger.log((0, _audit_events.scheduledReportAuditEvent)({
        action: _audit_events.ScheduledReportAuditAction.SCHEDULE,
        savedObject: {
          type: _saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE,
          id,
          name: jobParams.title
        },
        error: err
      }));
      return this.handleError(err, undefined, (_report = report) === null || _report === void 0 ? void 0 : _report.jobtype);
    }
  }
}
exports.ScheduleRequestHandler = ScheduleRequestHandler;