"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.retryOnError = void 0;
var _reportingCommon = require("@kbn/reporting-common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_DELAY_SECONDS = 30;
const delay = ms => new Promise(resolve => setTimeout(resolve, ms));
const retryOnError = async ({
  operation,
  retries,
  report,
  logger,
  attempt = 0
}) => {
  try {
    const result = await operation(report);
    if (attempt > 0) {
      logger.info(`Report generation for report[${report._id}] succeeded on attempt ${attempt + 1}.`);
    }
    return result;
  } catch (err) {
    // skip retry on certain errors
    if (err instanceof _reportingCommon.KibanaShuttingDownError) {
      throw err;
    }
    if (attempt < retries) {
      const retryCount = attempt + 1;
      const retryDelaySec = Math.min(Math.pow(2, retryCount), MAX_DELAY_SECONDS); // 2s, 4s, 8s, 16s, 30s, 30s, 30s...

      logger.warn(`Retrying report generation for report[${report._id}] after [${retryDelaySec}s] due to error: ${err.toString()} ${err.stack} - attempt ${retryCount} of ${retries + 1} failed.`);

      // delay with some randomness
      await delay(retryDelaySec + 1000 * Math.random());
      return retryOnError({
        operation,
        logger,
        report,
        retries,
        attempt: retryCount
      });
    }
    if (retries > 0) {
      // no retries left
      logger.error(`No retries left for report generation for report[${report._id}]. No report generated after ${retries + 1} attempts due to error: ${err.toString()} ${err.stack}`);
    }
    throw err;
  }
};
exports.retryOnError = retryOnError;