"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getWorkflowToolType = void 0;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _onechatServer = require("@kbn/onechat-server");
var _workflow = require("@kbn/onechat-common/tools/types/workflow");
var _prompts = require("@kbn/onechat-genai-utils/prompts");
var _execute_workflow = require("./execute_workflow");
var _generate_schema = require("./generate_schema");
var _schemas = require("./schemas");
var _validation = require("./validation");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getWorkflowToolType = ({
  workflowsManagement
}) => {
  // workflow plugin not present - we disable the workflow tool type
  if (!workflowsManagement) {
    return {
      toolType: _onechatCommon.ToolType.workflow,
      disabled: true
    };
  }
  return {
    toolType: _onechatCommon.ToolType.workflow,
    getDynamicProps: (config, {
      spaceId
    }) => {
      return {
        getHandler: () => {
          return async (params, {
            request
          }) => {
            const {
              management: workflowApi
            } = workflowsManagement;
            const workflowId = config.workflow_id;
            try {
              const workflowResults = await (0, _execute_workflow.executeWorkflow)({
                request,
                spaceId,
                workflowApi,
                workflowId,
                workflowParams: params,
                waitForCompletion: config.wait_for_completion
              });
              return {
                results: workflowResults
              };
            } catch (e) {
              return {
                results: [(0, _onechatServer.createErrorResult)({
                  message: `Error executing workflow: ${e}`,
                  metadata: {
                    workflowId
                  }
                })]
              };
            }
          };
        },
        getSchema: async () => {
          const workflow = await workflowsManagement.management.getWorkflow(config.workflow_id, spaceId);
          if (!workflow) {
            // if workflow is not accessible, error will be handled elsewhere
            return _zod.z.object({});
          }
          return (0, _generate_schema.generateSchema)({
            workflow
          });
        },
        getLlmDescription: ({
          description
        }) => {
          var _config$wait_for_comp;
          const wait = (_config$wait_for_comp = config.wait_for_completion) !== null && _config$wait_for_comp !== void 0 ? _config$wait_for_comp : true;
          const waitInstruction = wait ? `The tool will execute the workflow and then wait for it to complete up to ${_workflow.WAIT_FOR_COMPLETION_TIMEOUT_SEC}s` : 'The tool will execute the workflow and return immediately without waiting for its completion';
          return (0, _prompts.cleanPrompt)(`${description}

          ## Additional information
          - This tool executes the workflow with the ID '${config.workflow_id}'
          - ${waitInstruction}
          - If the workflow wasn't completed, a workflow execution ID will be returned.
          - The ${_onechatCommon.platformCoreTools.getWorkflowExecutionStatus} tool can be used later to check the status of the workflow execution

          `);
        }
      };
    },
    createSchema: _schemas.configurationSchema,
    updateSchema: _schemas.configurationUpdateSchema,
    validateForCreate: async ({
      config,
      context: {
        spaceId
      }
    }) => {
      await (0, _validation.validateWorkflowId)({
        workflows: workflowsManagement,
        workflowId: config.workflow_id,
        spaceId
      });
      return config;
    },
    validateForUpdate: async ({
      update,
      current,
      context: {
        spaceId
      }
    }) => {
      const mergedConfig = {
        ...current,
        ...update
      };
      await (0, _validation.validateWorkflowId)({
        workflows: workflowsManagement,
        workflowId: mergedConfig.workflow_id,
        spaceId
      });
      return mergedConfig;
    }
  };
};
exports.getWorkflowToolType = getWorkflowToolType;