"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.executeWorkflow = void 0;
var _v = require("@kbn/workflows/types/v1");
var _workflow = require("@kbn/onechat-common/tools/types/workflow");
var _workflows = require("@kbn/onechat-genai-utils/tools/utils/workflows");
var _results = require("@kbn/onechat-genai-utils/tools/utils/results");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const INITIAL_WAIT_MS = 1_000;
const CHECK_INTERVAL_MS = 2_500;
const finalStatuses = [_v.ExecutionStatus.COMPLETED, _v.ExecutionStatus.FAILED];
const executeWorkflow = async ({
  workflowId,
  workflowParams,
  request,
  spaceId,
  workflowApi,
  waitForCompletion = true,
  completionTimeoutSec = _workflow.WAIT_FOR_COMPLETION_TIMEOUT_SEC
}) => {
  const workflow = await workflowApi.getWorkflow(workflowId, spaceId);
  if (!workflow) {
    return [(0, _results.errorResult)(`Workflow '${workflowId}' not found.`)];
  }
  if (!workflow.enabled) {
    return [(0, _results.errorResult)(`Workflow '${workflowId}' is disabled and cannot be executed.`)];
  }
  if (!workflow.valid) {
    return [(0, _results.errorResult)(`Workflow '${workflowId}' has validation errors and cannot be executed.`)];
  }
  if (!workflow.definition) {
    return [(0, _results.errorResult)(`Workflow '${workflowId}' has no definition and cannot be executed.`)];
  }
  const executionId = await workflowApi.runWorkflow({
    id: workflow.id,
    name: workflow.name,
    enabled: workflow.enabled,
    definition: workflow.definition,
    yaml: workflow.yaml
  }, spaceId, workflowParams, request);
  const waitLimit = Date.now() + completionTimeoutSec * 1000;
  await waitMs(INITIAL_WAIT_MS);
  let execution;
  do {
    try {
      execution = await (0, _workflows.getExecutionState)({
        executionId,
        spaceId,
        workflowApi
      });
      const shouldReturn = waitForCompletion ? execution && finalStatuses.includes(execution.status) : execution;
      if (shouldReturn) {
        return [(0, _results.otherResult)({
          execution
        })];
      }
    } catch (e) {
      // trap - we just keep waiting until timeout
    }
    await waitMs(CHECK_INTERVAL_MS);
  } while (Date.now() < waitLimit);
  if (execution) {
    return [(0, _results.otherResult)({
      execution
    })];
  } else {
    return [(0, _results.errorResult)(`Workflow '${workflowId}' executed but execution not found after ${_workflow.WAIT_FOR_COMPLETION_TIMEOUT_SEC}s.`)];
  }
};
exports.executeWorkflow = executeWorkflow;
const waitMs = async durationMs => {
  await new Promise(resolve => setTimeout(resolve, durationMs));
};