"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.runTool = exports.runInternalTool = exports.createToolHandlerContext = void 0;
var _onechatCommon = require("@kbn/onechat-common");
var _inferenceTracing = require("@kbn/inference-tracing");
var _onechatServer = require("@kbn/onechat-server");
var _tools = require("@kbn/onechat-server/tools");
var _spaces = require("../../utils/spaces");
var _telemetry = require("../../telemetry");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const runTool = async ({
  toolExecutionParams,
  parentManager
}) => {
  const {
    trackingService,
    toolsService,
    request
  } = parentManager.deps;
  const {
    toolId,
    ...scopedParams
  } = toolExecutionParams;
  const toolRegistry = await toolsService.getRegistry({
    request
  });
  const tool = await toolRegistry.get(toolId);
  if (trackingService) {
    try {
      trackingService.trackToolCall(toolId, _telemetry.ToolCallSource.API);
    } catch (error) {
      /* empty */
    }
  }
  return runInternalTool({
    toolExecutionParams: {
      ...scopedParams,
      tool
    },
    parentManager
  });
};
exports.runTool = runTool;
const runInternalTool = async ({
  toolExecutionParams,
  parentManager
}) => {
  const {
    tool,
    toolParams
  } = toolExecutionParams;
  const context = (0, _utils.forkContextForToolRun)({
    parentContext: parentManager.context,
    toolId: tool.id
  });
  const manager = parentManager.createChild(context);
  const {
    resultStore
  } = manager.deps;
  const {
    results
  } = await (0, _inferenceTracing.withExecuteToolSpan)(tool.id, {
    tool: {
      input: toolParams
    }
  }, async () => {
    const schema = await tool.getSchema();
    const validation = schema.safeParse(toolParams);
    if (validation.error) {
      throw (0, _onechatCommon.createBadRequestError)(`Tool ${tool.id} was called with invalid parameters: ${validation.error.message}`);
    }
    const toolHandlerContext = await createToolHandlerContext({
      toolExecutionParams: {
        ...toolExecutionParams,
        toolId: tool.id
      },
      manager
    });
    try {
      const toolHandler = await tool.getHandler();
      return await toolHandler(validation.data, toolHandlerContext);
    } catch (err) {
      return {
        results: [(0, _onechatServer.createErrorResult)(err.message)]
      };
    }
  });
  const resultsWithIds = results.map(result => {
    var _result$tool_result_i;
    return {
      ...result,
      tool_result_id: (_result$tool_result_i = result.tool_result_id) !== null && _result$tool_result_i !== void 0 ? _result$tool_result_i : (0, _tools.getToolResultId)()
    };
  });
  resultsWithIds.forEach(result => {
    resultStore.add(result);
  });
  return {
    results: resultsWithIds
  };
};
exports.runInternalTool = runInternalTool;
const createToolHandlerContext = async ({
  manager,
  toolExecutionParams
}) => {
  const {
    onEvent
  } = toolExecutionParams;
  const {
    request,
    elasticsearch,
    spaces,
    modelProvider,
    toolsService,
    resultStore,
    logger
  } = manager.deps;
  const spaceId = (0, _spaces.getCurrentSpaceId)({
    request,
    spaces
  });
  return {
    request,
    spaceId,
    logger,
    esClient: elasticsearch.client.asScoped(request),
    modelProvider,
    runner: manager.getRunner(),
    toolProvider: (0, _utils.createToolProvider)({
      registry: await toolsService.getRegistry({
        request
      }),
      runner: manager.getRunner(),
      request
    }),
    resultStore: resultStore.asReadonly(),
    events: (0, _utils.createToolEventEmitter)({
      eventHandler: onEvent,
      context: manager.context
    })
  };
};
exports.createToolHandlerContext = createToolHandlerContext;