"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.relevanceSearchToolName = exports.naturalLanguageSearchToolName = exports.createRelevanceSearchTool = exports.createNaturalLanguageSearchTool = void 0;
var _zod = require("@kbn/zod");
var _inferenceTracing = require("@kbn/inference-tracing");
var _tools = require("@langchain/core/tools");
var _tools2 = require("@kbn/onechat-common/tools");
var _tools3 = require("@kbn/onechat-server/tools");
var _relevance_search = require("../relevance_search");
var _nl_search = require("../nl_search");
var _i18n = require("./i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const convertMatchResult = result => {
  return {
    tool_result_id: (0, _tools3.getToolResultId)(),
    type: _tools2.ToolResultType.resource,
    data: {
      reference: {
        id: result.id,
        index: result.index
      },
      partial: true,
      content: {
        highlights: result.highlights
      }
    }
  };
};
const relevanceSearchToolName = exports.relevanceSearchToolName = 'relevance_search';
const createRelevanceSearchTool = ({
  model,
  esClient,
  events
}) => {
  return (0, _tools.tool)(async ({
    term,
    index,
    size
  }) => {
    return (0, _inferenceTracing.withExecuteToolSpan)(relevanceSearchToolName, {
      tool: {
        input: {
          term,
          index,
          size
        }
      }
    }, async () => {
      events === null || events === void 0 ? void 0 : events.reportProgress(_i18n.progressMessages.performingRelevanceSearch({
        term
      }));
      const {
        results: rawResults
      } = await (0, _relevance_search.relevanceSearch)({
        target: index,
        term,
        size,
        model,
        esClient
      });
      const results = rawResults.map(convertMatchResult);
      const content = JSON.stringify(results);
      const artifact = {
        results
      };
      return [content, artifact];
    });
  }, {
    name: relevanceSearchToolName,
    responseFormat: 'content_and_artifact',
    schema: _zod.z.object({
      term: _zod.z.string().describe('Term to search for'),
      index: _zod.z.string().describe('Name of the index, alias or datastream to search against'),
      size: _zod.z.number().optional().default(10).describe('Number of documents to return. Defaults to 10.')
    }),
    description: `Find relevant documents in an index, alias or datastream, based on a simple fulltext search.

      Will search for documents performing a match query against the provided term(s).`
  });
};
exports.createRelevanceSearchTool = createRelevanceSearchTool;
const naturalLanguageSearchToolName = exports.naturalLanguageSearchToolName = 'natural_language_search';
const createNaturalLanguageSearchTool = ({
  model,
  esClient,
  events,
  logger,
  rowLimit,
  customInstructions
}) => {
  return (0, _tools.tool)(async ({
    query,
    index
  }) => {
    return (0, _inferenceTracing.withExecuteToolSpan)(naturalLanguageSearchToolName, {
      tool: {
        input: {
          query,
          index
        }
      }
    }, async () => {
      var _response$error;
      events === null || events === void 0 ? void 0 : events.reportProgress(_i18n.progressMessages.performingNlSearch({
        query
      }));
      const response = await (0, _nl_search.naturalLanguageSearch)({
        nlQuery: query,
        target: index,
        model,
        esClient,
        events,
        logger,
        rowLimit,
        customInstructions
      });
      const results = response.esqlData ? [{
        tool_result_id: (0, _tools3.getToolResultId)(),
        type: _tools2.ToolResultType.query,
        data: {
          esql: response.generatedQuery
        }
      }, {
        tool_result_id: (0, _tools3.getToolResultId)(),
        type: _tools2.ToolResultType.tabularData,
        data: {
          source: 'esql',
          query: response.generatedQuery,
          columns: response.esqlData.columns,
          values: response.esqlData.values
        }
      }] : [(0, _tools3.createErrorResult)({
        message: (_response$error = response.error) !== null && _response$error !== void 0 ? _response$error : 'Query was not executed',
        metadata: {
          query: response.generatedQuery
        }
      })];
      const content = JSON.stringify(results);
      const artifact = {
        results
      };
      return [content, artifact];
    });
  }, {
    name: naturalLanguageSearchToolName,
    responseFormat: 'content_and_artifact',
    schema: _zod.z.object({
      query: _zod.z.string().describe('A natural language query expressing the search request'),
      index: _zod.z.string().describe('Name of the index, alias or datastream to search against')
    }),
    description: `Given a natural language query, generate and execute the corresponding search request and returns the results in a tabular format

Example of natural language queries which can be passed to the tool:
  - "show me the last 5 documents from the index"
  - "what is the average order value?"
  - "list all products where status is 'in_stock' and price is less than 50"
  - "how many errors were logged in the past hour?"`
  });
};
exports.createNaturalLanguageSearchTool = createNaturalLanguageSearchTool;