"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OBSERVABILITY_GET_ANOMALY_DETECTION_JOBS_TOOL_ID = void 0;
exports.createGetAnomalyDetectionJobsTool = createGetAnomalyDetectionJobsTool;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const OBSERVABILITY_GET_ANOMALY_DETECTION_JOBS_TOOL_ID = exports.OBSERVABILITY_GET_ANOMALY_DETECTION_JOBS_TOOL_ID = 'observability.get_anomaly_detection_jobs';
const DEFAULT_JOBS_LIMIT = 10;
const DEFAULT_TIME_RANGE = {
  start: 'now-24h',
  end: 'now'
};
const getAnomalyDetectionJobsSchema = _zod.z.object({
  jobIds: _zod.z.array(_zod.z.string().min(1)).min(1).max(20).describe('Optional list of ML job IDs to query. Leave empty to include all anomaly detection jobs in this space.').optional(),
  limit: _zod.z.number().int().min(1).max(25).describe(`Maximum number of jobs to return. Defaults to ${DEFAULT_JOBS_LIMIT}.`).optional(),
  start: _zod.z.string().min(2).max(200).describe(`Start of the time range for anomaly records, expressed with Elasticsearch date math (e.g. now-1h). Defaults to ${DEFAULT_TIME_RANGE.start}.`).optional(),
  end: _zod.z.string().min(2).max(200).describe(`End of the time range for anomaly records, expressed with Elasticsearch date math. Defaults to ${DEFAULT_TIME_RANGE.end}.`).optional()
});
function createGetAnomalyDetectionJobsTool({
  core,
  plugins,
  logger
}) {
  const toolDefinition = {
    id: OBSERVABILITY_GET_ANOMALY_DETECTION_JOBS_TOOL_ID,
    type: _onechatCommon.ToolType.builtin,
    description: 'Return anomaly detection jobs and associated anomaly records. Useful for identifying unusual patterns in observability data.',
    schema: getAnomalyDetectionJobsSchema,
    tags: ['observability', 'machine_learning', 'anomaly_detection'],
    handler: async ({
      jobIds,
      limit: jobsLimit = DEFAULT_JOBS_LIMIT,
      start: rangeStart = DEFAULT_TIME_RANGE.start,
      end: rangeEnd = DEFAULT_TIME_RANGE.end
    }, {
      esClient,
      request
    }) => {
      const scopedEsClient = esClient.asCurrentUser;
      const mlClient = scopedEsClient.ml;
      try {
        const mlJobs = await getMlJobs({
          core,
          plugins,
          mlClient,
          request,
          logger,
          jobIds,
          jobsLimit,
          rangeStart,
          rangeEnd
        });
        if (!mlJobs.length) {
          return {
            results: [{
              type: _tool_result.ToolResultType.other,
              data: {
                jobs: [],
                totalReturned: 0,
                message: 'No anomaly detection jobs found for the provided filters.'
              }
            }]
          };
        }
        return {
          results: [{
            type: _tool_result.ToolResultType.other,
            data: {
              jobs: mlJobs,
              totalReturned: mlJobs.length
            }
          }]
        };
      } catch (error) {
        logger.error(`Error retrieving anomaly detection jobs: ${error.message}`);
        logger.debug(error);
        return {
          results: [{
            type: _tool_result.ToolResultType.error,
            data: {
              message: `Failed to retrieve anomaly detection jobs: ${error.message}`,
              stack: error.stack
            }
          }]
        };
      }
    }
  };
  return toolDefinition;
}
async function getMlJobs({
  core,
  plugins,
  mlClient,
  request,
  logger,
  jobIds = [],
  jobsLimit,
  rangeStart,
  rangeEnd
}) {
  var _plugins$ml;
  const [coreStart] = await core.getStartServices();
  const savedObjectsClient = coreStart.savedObjects.getScopedClient(request);
  const mlSystem = (_plugins$ml = plugins.ml) === null || _plugins$ml === void 0 ? void 0 : _plugins$ml.mlSystemProvider(request, savedObjectsClient);
  if (!mlSystem) {
    throw new Error('Machine Learning plugin is unavailable.');
  }
  const {
    jobs = []
  } = await mlClient.getJobs({
    job_id: jobIds.join(',')
  }).catch(error => {
    if (error.statusCode === 404) {
      return {
        jobs: []
      };
    }
    logger.error(`Error retrieving ML jobs: ${error.message}`);
    throw error;
  });
  return Promise.all(jobs.slice(0, jobsLimit).map(async job => {
    var _job$analysis_config, _job$datafeed_config, _job$analysis_config2, _job$analysis_config3;
    const topAnomalies = await getTopAnomalyRecords({
      mlSystem,
      jobId: job.job_id,
      start: rangeStart,
      end: rangeEnd
    });
    return {
      jobId: job.job_id,
      description: job.description,
      bucketSpan: (_job$analysis_config = job.analysis_config) === null || _job$analysis_config === void 0 ? void 0 : _job$analysis_config.bucket_span,
      datafeedIndices: (_job$datafeed_config = job.datafeed_config) === null || _job$datafeed_config === void 0 ? void 0 : _job$datafeed_config.indices,
      detectors: (_job$analysis_config2 = job.analysis_config) === null || _job$analysis_config2 === void 0 ? void 0 : (_job$analysis_config3 = _job$analysis_config2.detectors) === null || _job$analysis_config3 === void 0 ? void 0 : _job$analysis_config3.map(detector => ({
        description: detector.detector_description,
        function: detector.function,
        fieldName: detector.field_name
      })),
      topAnomalies
    };
  }));
}
async function getTopAnomalyRecords({
  mlSystem,
  jobId,
  start,
  end
}) {
  const response = await mlSystem.mlAnomalySearch({
    track_total_hits: false,
    size: 100,
    sort: [{
      record_score: {
        order: 'desc'
      }
    }],
    query: {
      bool: {
        filter: [{
          term: {
            job_id: jobId
          }
        }, {
          term: {
            result_type: 'record'
          }
        }, {
          term: {
            is_interim: false
          }
        }, {
          range: {
            timestamp: {
              gte: start,
              lte: end
            }
          }
        }]
      }
    },
    _source: ['timestamp', 'record_score', 'by_field_name', 'by_field_value', 'partition_field_name', 'partition_field_value', 'field_name', 'anomaly_score_explanation', 'typical', 'actual']
  }, [jobId]);
  const records = response.hits.hits.map(hit => hit._source).filter(record => record !== undefined);
  return records.map(record => ({
    timestamp: record.timestamp,
    anomalyScore: record.record_score,
    byFieldName: record.by_field_name,
    byFieldValue: record.by_field_value,
    partitionFieldName: record.partition_field_name,
    partitionFieldValue: record.partition_field_value,
    fieldName: record.field_name,
    anomalyScoreExplanation: record.anomaly_score_explanation,
    typicalValue: record.typical,
    actualValue: record.actual
  }));
}