"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OBSERVABILITY_GET_ALERTS_TOOL_ID = void 0;
exports.createGetAlertsTool = createGetAlertsTool;
exports.defaultFields = void 0;
var _zod = require("@kbn/zod");
var _lodash = require("lodash");
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _technical_rule_data_field_names = require("@kbn/rule-registry-plugin/common/technical_rule_data_field_names");
var _constants = require("@kbn/observability-plugin/common/constants");
var _get_relevant_alert_fields = require("./get_relevant_alert_fields");
var _get_hits_total = require("../../utils/get_hits_total");
var _dsl_filters = require("../../utils/dsl_filters");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const OBSERVABILITY_GET_ALERTS_TOOL_ID = exports.OBSERVABILITY_GET_ALERTS_TOOL_ID = 'observability.get_alerts';
const defaultFields = exports.defaultFields = ['@timestamp', 'kibana.alert.start', 'kibana.alert.end', 'kibana.alert.flapping', 'kibana.alert.group', 'kibana.alert.instance.id', 'kibana.alert.reason', 'kibana.alert.rule.category', 'kibana.alert.rule.name', 'kibana.alert.rule.tags', 'kibana.alert.start', 'kibana.alert.status', 'kibana.alert.time_range.gte', 'kibana.alert.time_range.lte', 'kibana.alert.workflow_status', 'tags',
// infra
'host.name', 'container.id', 'kubernetes.pod.name',
// APM
'processor.event', 'service.environment', 'service.name', 'service.node.name', 'transaction.type', 'transaction.name'];
const OMITTED_ALERT_FIELDS = ['event.action', 'event.kind', 'kibana.alert.rule.execution.uuid', 'kibana.alert.rule.revision', 'kibana.alert.rule.tags', 'kibana.alert.rule.uuid', 'kibana.alert.workflow_status', 'kibana.space_ids', 'kibana.alert.time_range', 'kibana.version'];
const getAlertsSchema = _zod.z.object({
  start: _zod.z.string().describe('The start of the time range, in Elasticsearch date math, like `now-24h`.').min(1),
  end: _zod.z.string().describe('The end of the time range, in Elasticsearch date math, like `now`.').min(1),
  query: _zod.z.string().min(1).describe('Natural language query to guide relevant field selection.'),
  kqlFilter: _zod.z.string().optional().describe('Filter alerts by field:value pairs'),
  includeRecovered: _zod.z.boolean().optional().describe('Whether to include recovered/closed alerts. Defaults to false, which means only active alerts will be returned.')
});
function createGetAlertsTool({
  core,
  logger
}) {
  const toolDefinition = {
    id: OBSERVABILITY_GET_ALERTS_TOOL_ID,
    type: _onechatCommon.ToolType.builtin,
    description: `Retrieve Observability alerts and relevant fields for a given time range. Defaults to active alerts (set includeRecovered to true to include recovered alerts).`,
    schema: getAlertsSchema,
    tags: ['observability', 'alerts'],
    handler: async ({
      start,
      end,
      kqlFilter,
      includeRecovered,
      query
    }, handlerinfo) => {
      try {
        const [coreStart, pluginStart] = await core.getStartServices();
        const alertsClient = await pluginStart.ruleRegistry.getRacClientWithRequest(handlerinfo.request);
        const selectedFields = await (0, _get_relevant_alert_fields.getRelevantAlertFields)({
          coreStart,
          pluginStart,
          request: handlerinfo.request,
          modelProvider: handlerinfo.modelProvider,
          logger,
          query
        });
        const response = await alertsClient.find({
          ruleTypeIds: _constants.OBSERVABILITY_RULE_TYPE_IDS_WITH_SUPPORTED_STACK_RULE_TYPES,
          consumers: [_technical_rule_data_field_names.AlertConsumers.APM, _technical_rule_data_field_names.AlertConsumers.INFRASTRUCTURE, _technical_rule_data_field_names.AlertConsumers.LOGS, _technical_rule_data_field_names.AlertConsumers.UPTIME, _technical_rule_data_field_names.AlertConsumers.SLO, _technical_rule_data_field_names.AlertConsumers.OBSERVABILITY, _technical_rule_data_field_names.AlertConsumers.ALERTS],
          query: {
            bool: {
              filter: [{
                range: {
                  'kibana.alert.start': {
                    gte: start,
                    lte: end
                  }
                }
              }, ...(0, _dsl_filters.kqlFilter)(kqlFilter), ...(includeRecovered ? [] : [{
                term: {
                  [_technical_rule_data_field_names.ALERT_STATUS]: _technical_rule_data_field_names.ALERT_STATUS_ACTIVE
                }
              }])]
            }
          },
          size: 10
        });
        const total = (0, _get_hits_total.getHitsTotal)(response);
        const alerts = response.hits.hits.map(hit => {
          var _hit$_source;
          return (0, _lodash.omit)((_hit$_source = hit._source) !== null && _hit$_source !== void 0 ? _hit$_source : {}, ...OMITTED_ALERT_FIELDS);
        });
        return {
          results: [{
            type: _tool_result.ToolResultType.other,
            data: {
              total,
              alerts,
              selectedFields: selectedFields.length === 0 ? defaultFields : selectedFields
            }
          }]
        };
      } catch (error) {
        logger.error(`Error fetching observability alerts: ${error.message}`);
        logger.debug(error);
        return {
          results: [{
            type: _tool_result.ToolResultType.error,
            data: {
              message: `Failed to fetch observability alerts: ${error.message}`,
              stack: error.stack
            }
          }]
        };
      }
    }
  };
  return toolDefinition;
}