"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isOnlySingleFeatureType = isOnlySingleFeatureType;
exports.pluckCategoricalStyleMetaFromFeatures = pluckCategoricalStyleMetaFromFeatures;
exports.pluckStyleMetaFromFeatures = pluckStyleMetaFromFeatures;
var _constants = require("../../../../../common/constants");
var _dynamic_style_property = require("../../../styles/vector/properties/dynamic_style_property");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const POINTS = [_constants.GEO_JSON_TYPE.POINT, _constants.GEO_JSON_TYPE.MULTI_POINT];
const LINES = [_constants.GEO_JSON_TYPE.LINE_STRING, _constants.GEO_JSON_TYPE.MULTI_LINE_STRING];
const POLYGONS = [_constants.GEO_JSON_TYPE.POLYGON, _constants.GEO_JSON_TYPE.MULTI_POLYGON];
async function pluckStyleMetaFromFeatures(features, supportedShapeTypes, dynamicProperties) {
  const hasFeatureType = {
    [_constants.VECTOR_SHAPE_TYPE.POINT]: false,
    [_constants.VECTOR_SHAPE_TYPE.LINE]: false,
    [_constants.VECTOR_SHAPE_TYPE.POLYGON]: false
  };
  if (supportedShapeTypes.length > 1) {
    for (let i = 0; i < features.length; i++) {
      var _feature$properties;
      const feature = features[i];

      // ignore centroid features as they are added for styling and not part of the real data set
      if ((_feature$properties = feature.properties) !== null && _feature$properties !== void 0 && _feature$properties[_constants.KBN_IS_CENTROID_FEATURE]) {
        continue;
      }
      if (!hasFeatureType[_constants.VECTOR_SHAPE_TYPE.POINT] && POINTS.includes(feature.geometry.type)) {
        hasFeatureType[_constants.VECTOR_SHAPE_TYPE.POINT] = true;
      }
      if (!hasFeatureType[_constants.VECTOR_SHAPE_TYPE.LINE] && LINES.includes(feature.geometry.type)) {
        hasFeatureType[_constants.VECTOR_SHAPE_TYPE.LINE] = true;
      }
      if (!hasFeatureType[_constants.VECTOR_SHAPE_TYPE.POLYGON] && POLYGONS.includes(feature.geometry.type)) {
        hasFeatureType[_constants.VECTOR_SHAPE_TYPE.POLYGON] = true;
      }
    }
  }
  const styleMeta = {
    geometryTypes: {
      isPointsOnly: isOnlySingleFeatureType(_constants.VECTOR_SHAPE_TYPE.POINT, supportedShapeTypes, hasFeatureType),
      isLinesOnly: isOnlySingleFeatureType(_constants.VECTOR_SHAPE_TYPE.LINE, supportedShapeTypes, hasFeatureType),
      isPolygonsOnly: isOnlySingleFeatureType(_constants.VECTOR_SHAPE_TYPE.POLYGON, supportedShapeTypes, hasFeatureType)
    },
    fieldMeta: {}
  };
  if (dynamicProperties.length === 0 || features.length === 0) {
    // no additional meta data to pull from source data request.
    return styleMeta;
  }
  dynamicProperties.forEach(dynamicProperty => {
    const name = dynamicProperty.getFieldName();
    if (!styleMeta.fieldMeta[name]) {
      styleMeta.fieldMeta[name] = {
        categories: []
      };
    }
    const categories = pluckCategoricalStyleMetaFromFeatures(dynamicProperty, features);
    if (categories.length) {
      styleMeta.fieldMeta[name].categories = categories;
    }
    const ordinalStyleMeta = pluckOrdinalStyleMetaFromFeatures(dynamicProperty, features);
    if (ordinalStyleMeta) {
      styleMeta.fieldMeta[name].range = ordinalStyleMeta;
    }
  });
  return styleMeta;
}
function pluckOrdinalStyleMetaFromFeatures(property, features) {
  const field = property.getField();
  if (!field || !property.isOrdinal()) {
    return null;
  }
  const name = property.getFieldName();
  const isCount = field.isCount();
  let min = isCount ? 1 : Infinity;
  let max = -Infinity;
  for (let i = 0; i < features.length; i++) {
    const feature = features[i];
    const newValue = feature.properties ? parseFloat(feature.properties[name]) : NaN;
    if (!isNaN(newValue)) {
      if (!isCount) {
        min = Math.min(min, newValue);
      }
      max = Math.max(max, newValue);
    }
  }
  return min === Infinity || max === -Infinity ? null : {
    min,
    max,
    delta: max - min
  };
}
function pluckCategoricalStyleMetaFromFeatures(property, features) {
  const size = property.getNumberOfCategories();
  if (!property.isCategorical() || size <= 0) {
    return [];
  }
  const counts = new Map();
  for (let i = 0; i < features.length; i++) {
    const feature = features[i];
    const term = feature.properties ? feature.properties[property.getFieldName()] : undefined;
    // properties object may be sparse, so need to check if the field is effectively present
    if (typeof term !== undefined) {
      if (counts.has(term)) {
        counts.set(term, counts.get(term) + 1);
      } else {
        counts.set(term, 1);
      }
    }
  }
  const ordered = [];
  for (const [key, value] of counts) {
    ordered.push({
      key,
      count: value
    });
  }
  ordered.sort((a, b) => {
    return b.count - a.count;
  });
  if (ordered.length <= size) {
    return ordered;
  }
  const topCategories = ordered.slice(0, size);
  let otherCategoryCount = 0;
  for (let i = size; i < ordered.length; i++) {
    otherCategoryCount += ordered[i].count;
  }
  return [...topCategories, {
    key: _dynamic_style_property.OTHER_CATEGORY_KEY,
    count: otherCategoryCount
  }];
}
function isOnlySingleFeatureType(featureType, supportedShapeTypes, hasFeatureType) {
  if (supportedShapeTypes.length === 1) {
    return supportedShapeTypes[0] === featureType;
  }
  const featureTypes = Object.keys(hasFeatureType);
  // @ts-expect-error
  return featureTypes.reduce((accumulator, featureTypeKey) => {
    const hasFeature = hasFeatureType[featureTypeKey];
    return featureTypeKey === featureType ? accumulator && hasFeature : accumulator && !hasFeature;
  }, true);
}