"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildDatasourceStates = exports.addLayerFormulaColumns = exports.addLayerColumn = void 0;
exports.buildReferences = buildReferences;
exports.extractReferences = extractReferences;
exports.getAdhocDataviews = void 0;
exports.getDataView = getDataView;
exports.getDatasetIndex = getDatasetIndex;
exports.getDefaultReferences = void 0;
exports.isDataViewDataset = isDataViewDataset;
exports.isEsqlTableTypeDataset = isEsqlTableTypeDataset;
exports.isLensAPIFormat = isLensAPIFormat;
exports.isLensLegacyAttributes = isLensLegacyAttributes;
exports.isLensLegacyFormat = isLensLegacyFormat;
exports.isSingleLayer = isSingleLayer;
exports.mapToFormula = mapToFormula;
var _uuid = require("uuid");
var _esqlUtils = require("@kbn/esql-utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const getDefaultReferences = (index, dataLayerId) => {
  return [{
    type: 'index-pattern',
    id: index,
    name: `indexpattern-datasource-layer-${dataLayerId}`
  }];
};
exports.getDefaultReferences = getDefaultReferences;
function mapToFormula(layer) {
  const {
    label,
    decimals,
    format,
    compactValues: compact,
    normalizeByUnit,
    value
  } = layer;
  const formulaFormat = format ? {
    id: format,
    params: {
      decimals: decimals !== null && decimals !== void 0 ? decimals : 2,
      ...(!!compact ? {
        compact
      } : undefined)
    }
  } : undefined;
  return {
    formula: value,
    label,
    timeScale: normalizeByUnit,
    format: formulaFormat
  };
}
function extractReferences(dataviews) {
  const adHocDataViews = getAdhocDataviews(dataviews);
  return {
    ...buildReferences(dataviews, adHocDataViews),
    adHocDataViews
  };
}
function buildReferences(dataviews, adHocDataViews) {
  const references = [];
  const internalReferences = [];
  for (const [layerId, dataview] of Object.entries(dataviews)) {
    if (dataview.id) {
      const defaultRefs = getDefaultReferences(dataview.id, layerId);
      if (adHocDataViews[dataview.id]) {
        internalReferences.push(...defaultRefs);
      } else {
        references.push(...defaultRefs);
      }
    }
  }
  return {
    references: references.flat(),
    internalReferences: internalReferences.flat()
  };
}
const getAdhocDataView = dataView => {
  var _dataView$id;
  return {
    [(_dataView$id = dataView.id) !== null && _dataView$id !== void 0 ? _dataView$id : (0, _uuid.v4)()]: {
      ...dataView.toSpec(false)
    }
  };
};

// Getting the spec from a data view is a heavy operation, that's why the result is cached.
const getAdhocDataviews = dataviews => {
  let adHocDataViews = {};
  [...Array.from(new Set(Object.values(dataviews)))].forEach(d => {
    adHocDataViews = {
      ...adHocDataViews,
      ...getAdhocDataView(d)
    };
  });
  return adHocDataViews;
};
exports.getAdhocDataviews = getAdhocDataviews;
function isSingleLayer(layer) {
  return layer && typeof layer === 'object' && ('columnOrder' in layer || 'columns' in layer);
}

/**
 * Retrieves an existing data view by its title (index) or creates an Ad-Hoc Dataview if it does not exist.
 *
 * Behavior:
 * - If an explicit `id` is provided, a new ad-hoc data view is picked from the cache or created with that `id`.
 * - If no `id` is provided, the function first attempts to retrieve an existing data view whose id matches `index`.
 *   - If retrieval succeeds, the existing data view is returned.
 *   - If retrieval fails (e.g. it does not exist), a new data view is created using `index` as the title.
 *
 * @param params.index The index pattern or title to use for lookup or creation.
 * @param params.timeFieldName The name of the time field to associate with the data view.
 * @param dataViewsAPI The DataViews service API used to get or create data views.
 * @param id Optional explicit id to assign to the data view; if provided, creation is forced.
 * @returns A promise resolving to the retrieved or newly created data view.
 * @throws Re-throws any error coming from the underlying DataViews API when creation fails.
 */
async function getDataView({
  index,
  timeFieldName
}, dataViewsAPI, id) {
  if (id) {
    return dataViewsAPI.create({
      id,
      title: index,
      timeFieldName
    });
  }
  try {
    return await dataViewsAPI.get(index, false);
  } catch {
    return dataViewsAPI.create({
      title: index,
      timeFieldName
    });
  }
}
function getDatasetIndex(dataset) {
  if (isDataViewDataset(dataset)) {
    var _dataset$timeFieldNam;
    return {
      index: dataset.index,
      timeFieldName: (_dataset$timeFieldNam = dataset.timeFieldName) !== null && _dataset$timeFieldNam !== void 0 ? _dataset$timeFieldNam : '@timestamp'
    };
  } else if (isESQLDataset(dataset)) {
    return {
      index: (0, _esqlUtils.getIndexPatternFromESQLQuery)(dataset.esql),
      timeFieldName: '@timestamp'
    };
  }
}
function buildDatasourceStatesLayer(layer, i, dataset, dataView, buildFormulaLayers, getValueColumns) {
  function buildValueLayer(config) {
    const table = dataset;
    const newLayer = {
      table,
      columns: getValueColumns(layer, i),
      allColumns: table.columns.map(column => ({
        fieldName: column.name,
        columnId: column.id,
        meta: column.meta
      })),
      index: '',
      query: undefined
    };
    return newLayer;
  }
  function buildESQLLayer(config) {
    const columns = getValueColumns(layer, i);
    const newLayer = {
      index: dataView.id,
      query: {
        esql: dataset.esql
      },
      timeField: dataView.timeFieldName,
      columns,
      allColumns: columns
    };
    return newLayer;
  }
  if (isESQLDataset(dataset)) {
    return ['textBased', buildESQLLayer(layer)];
  } else if ('type' in dataset) {
    return ['textBased', buildValueLayer(layer)];
  }
  return ['formBased', buildFormulaLayers(layer, i, dataView)];
}
const buildDatasourceStates = async (config, dataviews, buildFormulaLayers, getValueColumns, dataViewsAPI) => {
  let layers = {};
  const mainDataset = config.dataset;
  const configLayers = 'layers' in config ? config.layers : [config];
  for (let i = 0; i < configLayers.length; i++) {
    var _layer$dataset;
    const layer = configLayers[i];
    const layerId = `layer_${i}`;
    const dataset = (_layer$dataset = layer.dataset) !== null && _layer$dataset !== void 0 ? _layer$dataset : mainDataset;
    if (!dataset && 'type' in layer && layer.type !== 'annotation') {
      throw Error('dataset must be defined');
    }
    if (dataset) {
      const index = getDatasetIndex(dataset);
      const dataView = index ? await getDataView(index, dataViewsAPI, isESQLDataset(dataset) ? JSON.stringify(index) : undefined) : undefined;
      const [type, layerConfig] = buildDatasourceStatesLayer(layer, i, dataset, dataView, buildFormulaLayers, getValueColumns);
      if (layerConfig) {
        var _layers$type;
        layers = {
          ...layers,
          [type]: {
            layers: isSingleLayer(layerConfig) ? {
              ...((_layers$type = layers[type]) === null || _layers$type === void 0 ? void 0 : _layers$type.layers),
              [layerId]: layerConfig
            } :
            // metric chart can return 2 layers (one for the metric and one for the trendline)
            {
              ...layerConfig
            }
          }
        };
      }
      if (dataView) {
        var _layers$type$layers, _layers$type2;
        Object.keys((_layers$type$layers = (_layers$type2 = layers[type]) === null || _layers$type2 === void 0 ? void 0 : _layers$type2.layers) !== null && _layers$type$layers !== void 0 ? _layers$type$layers : []).forEach(id => {
          dataviews[id] = dataView;
        });
      }
    }
  }
  return layers;
};
exports.buildDatasourceStates = buildDatasourceStates;
const addLayerColumn = (layer, columnName, config, first = false) => {
  layer.columns = {
    ...layer.columns,
    [columnName]: config
  };
  if (first) {
    layer.columnOrder.unshift(columnName);
  } else {
    layer.columnOrder.push(columnName);
  }
};
exports.addLayerColumn = addLayerColumn;
const addLayerFormulaColumns = (layer, columns, postfix = '') => {
  const altObj = Object.fromEntries(Object.entries(columns.columns).map(([key, value]) =>
  // Modify key here
  [`${key}${postfix}`, value]));
  layer.columns = {
    ...layer.columns,
    ...altObj
  };
  layer.columnOrder.push(...columns.columnOrder.map(c => `${c}${postfix}`));
};
exports.addLayerFormulaColumns = addLayerFormulaColumns;
function isESQLDataset(dataset) {
  return 'esql' in dataset;
}
function isDataViewDataset(dataset) {
  return 'index' in dataset;
}
function isLensAPIFormat(config) {
  return typeof config === 'object' && config !== null && 'type' in config;
}
function isLensLegacyFormat(config) {
  return typeof config === 'object' && config !== null && 'chartType' in config;
}
function isLensLegacyAttributes(config) {
  return typeof config === 'object' && config !== null && 'state' in config && 'references' in config;
}
function isEsqlTableTypeDataset(dataset) {
  return dataset.type === 'esql' || dataset.type === 'table';
}