"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildFormBasedXYLayer = buildFormBasedXYLayer;
exports.buildXYLayer = buildXYLayer;
exports.getValueColumns = getValueColumns;
var _esql_column = require("../../columns/esql_column");
var _utils = require("../../utils");
var _helpers = require("./helpers");
var _metric = require("../../columns/metric");
var _buckets = require("../../columns/buckets");
var _coloring = require("../../coloring");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const X_ACCESSOR = 'x';
const BREAKDOWN_ACCESSOR = 'breakdown';
const METRIC_ACCESSOR_PREFIX = 'y';
const REFERENCE_LINE_ACCESSOR_PREFIX = 'threshold';
function getValueColumns(layer, i) {
  if (!(0, _helpers.isAPIXYLayer)(layer) || !(0, _helpers.isAPIesqlXYLayer)(layer)) {
    return [];
  }
  if ((0, _helpers.isAPIAnnotationLayer)(layer)) {
    return [];
  }
  if ((0, _helpers.isAPIReferenceLineLayer)(layer)) {
    return [...layer.thresholds.map((t, index) => (0, _esql_column.getValueColumn)(`referenceLine${index}`, t.column, 'number'))];
  }
  return [...(layer.x ? [(0, _esql_column.getValueColumn)((0, _helpers.getAccessorNameForXY)(layer, X_ACCESSOR), layer.x.column)] : []), ...layer.y.map((y, index) => (0, _esql_column.getValueColumn)((0, _helpers.getAccessorNameForXY)(layer, METRIC_ACCESSOR_PREFIX, index), y.column, 'number')), ...(layer.breakdown_by ? [(0, _esql_column.getValueColumn)((0, _helpers.getAccessorNameForXY)(layer, BREAKDOWN_ACCESSOR), layer.breakdown_by.column)] : [])];
}
function buildDataLayer(layer, i) {
  const seriesTypeLabel = layer.type.includes('percentage') ? `${layer.type}_stacked` : layer.type;
  const yConfig = layer.y.map((yMetric, index) => {
    var _yMetric$color, _yMetric$color2;
    return {
      ...((_yMetric$color = yMetric.color) !== null && _yMetric$color !== void 0 && _yMetric$color.color ? {
        color: (_yMetric$color2 = yMetric.color) === null || _yMetric$color2 === void 0 ? void 0 : _yMetric$color2.color
      } : {}),
      ...(yMetric.axis ? {
        axisMode: yMetric.axis
      } : {}),
      forAccessor: (0, _helpers.getAccessorNameForXY)(layer, METRIC_ACCESSOR_PREFIX, index)
    };
  });
  const meaningFulYConfig = yConfig.filter(y => Object.values(y).length > 1);
  return {
    layerId: (0, _helpers.getIdForLayer)(layer, i),
    accessors: yConfig.map(({
      forAccessor
    }) => forAccessor),
    layerType: 'data',
    seriesType: seriesTypeLabel,
    ...(layer.x ? {
      xAccessor: (0, _helpers.getAccessorNameForXY)(layer, X_ACCESSOR)
    } : {}),
    ...(meaningFulYConfig.length ? {
      yConfig: meaningFulYConfig
    } : {}),
    ...(layer.breakdown_by ? {
      splitAccessor: (0, _helpers.getAccessorNameForXY)(layer, BREAKDOWN_ACCESSOR)
    } : {}),
    ...(layer.breakdown_by && 'collapse_by' in layer.breakdown_by ? {
      collapseFn: layer.breakdown_by.collapse_by
    } : {}),
    ...(layer.breakdown_by && 'color' in layer.breakdown_by ? {
      colorMapping: (0, _coloring.fromColorMappingAPIToLensState)(layer.breakdown_by.color)
    } : {})
  };
}
function buildAnnotationLayer(layer, i, dataViewId) {
  return {
    layerType: 'annotations',
    layerId: (0, _helpers.getIdForLayer)(layer, i),
    indexPatternId: dataViewId,
    ignoreGlobalFilters: layer.ignore_global_filters,
    annotations: layer.events.map((annotation, index) => {
      var _annotation$label3, _annotation$color3, _annotation$text;
      if (annotation.type === 'range') {
        var _annotation$color, _annotation$label;
        return {
          type: 'manual',
          id: `${layer.type}_event_${index}`,
          key: {
            type: 'range',
            timestamp: String(annotation.interval.from),
            endTimestamp: String(annotation.interval.to)
          },
          outside: annotation.fill === 'outside',
          color: (_annotation$color = annotation.color) === null || _annotation$color === void 0 ? void 0 : _annotation$color.color,
          label: (_annotation$label = annotation.label) !== null && _annotation$label !== void 0 ? _annotation$label : 'Event',
          ...(annotation.hidden != null ? {
            hidden: annotation.hidden
          } : {})
        };
      }
      if (annotation.type === 'point') {
        var _annotation$color2, _annotation$label2;
        return {
          type: 'manual',
          id: `${layer.type}_event_${index}`,
          key: {
            type: 'point_in_time',
            timestamp: String(annotation.timestamp)
          },
          color: (_annotation$color2 = annotation.color) === null || _annotation$color2 === void 0 ? void 0 : _annotation$color2.color,
          label: (_annotation$label2 = annotation.label) !== null && _annotation$label2 !== void 0 ? _annotation$label2 : 'Event',
          ...(annotation.hidden != null ? {
            hidden: annotation.hidden
          } : {}),
          ...(annotation.text != null ? {
            textVisibility: annotation.text === 'label'
          } : {})
        };
      }
      return {
        type: 'query',
        id: `${layer.type}_event_${index}`,
        filter: {
          type: 'kibana_query',
          ...annotation.query
        },
        label: (_annotation$label3 = annotation.label) !== null && _annotation$label3 !== void 0 ? _annotation$label3 : 'Event',
        color: (_annotation$color3 = annotation.color) === null || _annotation$color3 === void 0 ? void 0 : _annotation$color3.color,
        ...(annotation.hidden != null ? {
          hidden: annotation.hidden
        } : {}),
        timeField: annotation.time_field,
        ...(annotation.extra_fields ? {
          extraFields: annotation.extra_fields
        } : {}),
        ...(annotation.text != null ? {
          textVisibility: annotation.text === 'label'
        } : {}),
        ...(typeof annotation.text !== 'string' && ((_annotation$text = annotation.text) === null || _annotation$text === void 0 ? void 0 : _annotation$text.type) === 'field' ? {
          textField: annotation.text.field
        } : {}),
        key: {
          type: 'point_in_time'
        }
      };
    })
  };
}
function buildReferenceLineLayer(layer, i) {
  const yConfig = layer.thresholds.map((threshold, index) => {
    var _threshold$color;
    return {
      icon: threshold.icon,
      lineWidth: threshold.stroke_width,
      lineStyle: threshold.stroke_dash,
      textVisibility: threshold.text ? threshold.text === 'label' : undefined,
      fill: threshold.fill,
      color: (_threshold$color = threshold.color) === null || _threshold$color === void 0 ? void 0 : _threshold$color.color,
      axisMode: threshold.axis,
      forAccessor: (0, _helpers.getAccessorNameForXY)(layer, REFERENCE_LINE_ACCESSOR_PREFIX, index)
    };
  });
  return {
    layerType: 'referenceLine',
    layerId: (0, _helpers.getIdForLayer)(layer, i),
    yConfig,
    accessors: yConfig.map(({
      forAccessor
    }) => forAccessor)
  };
}
function buildXYLayer(layer, i, dataViewId) {
  if (!(0, _helpers.isAPIXYLayer)(layer)) {
    return;
  }

  // now enrich the layer based on its type
  if ((0, _helpers.isAPIAnnotationLayer)(layer)) {
    return buildAnnotationLayer(layer, i, dataViewId);
  }
  if ((0, _helpers.isAPIReferenceLineLayer)(layer)) {
    return buildReferenceLineLayer(layer, i);
  }
  return buildDataLayer(layer, i);
}
function buildFormBasedXYLayer(layer, i) {
  // annotation layer have no datasource state
  if (!(0, _helpers.isAPIXYLayer)(layer) || (0, _helpers.isAPIesqlXYLayer)(layer) || (0, _helpers.isAPIAnnotationLayer)(layer)) {
    return {};
  }
  const layerId = (0, _helpers.getIdForLayer)(layer, i);
  const datasource = (0, _utils.generateLayer)(layerId, layer);
  const newLayer = datasource[layerId];
  if ((0, _helpers.isAPIReferenceLineLayer)(layer)) {
    for (const [index, column] of Object.entries(layer.thresholds)) {
      const columns = (0, _metric.fromMetricAPItoLensState)(column);
      (0, _utils.addLayerColumn)(newLayer, (0, _helpers.getAccessorNameForXY)(layer, REFERENCE_LINE_ACCESSOR_PREFIX, Number(index)), columns);
    }
  }
  if ((0, _helpers.isAPIDataLayer)(layer)) {
    // convert metrics in buckets, do not flat yet
    const yColumnsConverted = layer.y.map(_metric.fromMetricAPItoLensState);
    const yColumnsWithIds = [];
    // now fix the ids of referenced columns
    for (const [index, convertedColumns] of Object.entries(yColumnsConverted)) {
      const [mainMetric, refMetric] = convertedColumns;
      const id = (0, _helpers.getAccessorNameForXY)(layer, METRIC_ACCESSOR_PREFIX, Number(index));
      yColumnsWithIds.push({
        column: mainMetric,
        id
      });
      if (refMetric) {
        // Use a different format for reference column ids
        // as visualization doesn't know about them, so wrong id could be generated on that side
        const refId = (0, _helpers.getAccessorNameForXY)(layer, `${METRIC_ACCESSOR_PREFIX}_ref`, Number(index));
        // rewrite the mainMetric reference id to match the newly generated one
        if ('references' in mainMetric && Array.isArray(mainMetric.references)) {
          mainMetric.references = [refId];
        }
        yColumnsWithIds.push({
          column: refMetric,
          id: refId
        });
      }
    }
    const xColumns = layer.x ? (0, _buckets.fromBucketLensApiToLensState)(layer.x, yColumnsWithIds) : undefined;
    const breakdownColumns = layer.breakdown_by ? (0, _buckets.fromBucketLensApiToLensState)(layer.breakdown_by, yColumnsWithIds) : undefined;

    // Add bucketed coluns first
    if (xColumns) {
      (0, _utils.addLayerColumn)(newLayer, (0, _helpers.getAccessorNameForXY)(layer, X_ACCESSOR), xColumns);
    }
    if (breakdownColumns) {
      var _layer$breakdown_by;
      const breakdownById = (0, _helpers.getAccessorNameForXY)(layer, BREAKDOWN_ACCESSOR);
      (0, _utils.addLayerColumn)(newLayer, breakdownById, breakdownColumns, ((_layer$breakdown_by = layer.breakdown_by) === null || _layer$breakdown_by === void 0 ? void 0 : _layer$breakdown_by.aggregate_first) || !layer.x);
    }

    // then metric ones
    for (const {
      id,
      column
    } of yColumnsWithIds) {
      (0, _utils.addLayerColumn)(newLayer, id, column);
    }
  }
  return datasource;
}