"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useSessionAgentListState = exports.defaultAgentListState = void 0;
var _react = require("react");
var _useSessionStorage = _interopRequireDefault(require("react-use/lib/useSessionStorage"));
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const FLEET_NAMESPACE = 'fleet';
const AGENT_LIST_SESSION_STORAGE_KEY = 'agentListState';
const defaultAgentListState = exports.defaultAgentListState = {
  search: '',
  selectedAgentPolicies: [],
  selectedStatus: ['healthy', 'unhealthy', 'orphaned', 'updating', 'offline'],
  selectedTags: [],
  showUpgradeable: false,
  sort: {
    field: 'enrolled_at',
    direction: 'desc'
  },
  page: {
    index: 0,
    size: 20
  }
};
const useSessionAgentListState = () => {
  const fullStorageKey = `${FLEET_NAMESPACE}.${AGENT_LIST_SESSION_STORAGE_KEY}`;
  const [sessionState, setSessionState] = (0, _useSessionStorage.default)(fullStorageKey, defaultAgentListState);

  // Utility functions
  const updateTableState = (0, _react.useCallback)(partialState => {
    const newState = {
      ...sessionState,
      ...partialState
    };
    setSessionState(newState);
  }, [sessionState, setSessionState]);

  // Atomic update function for table changes - prevents multiple rapid updates
  const onTableChange = (0, _react.useCallback)(changes => {
    const latestState = sessionState;
    const updates = {};

    // Check if pagination has actually changed using isMatch for partial deep compare
    if (changes.page && !(0, _lodash.isMatch)(latestState.page, changes.page)) {
      updates.page = {
        index: changes.page.index,
        size: changes.page.size
      };
    }

    // Check if sort has actually changed using isMatch for partial deep compare
    if (changes.sort && !(0, _lodash.isMatch)(latestState.sort, changes.sort)) {
      updates.sort = {
        field: changes.sort.field,
        direction: changes.sort.direction
      };
    }

    // Only update if there are actual changes
    if (Object.keys(updates).length > 0) {
      const newState = {
        ...latestState,
        ...updates
      };
      setSessionState(newState);
    }
  }, [sessionState, setSessionState]);

  // Reset filters to default state
  // Do not reset sort parameters, but reset page back to index 0
  const clearFilters = (0, _react.useCallback)(() => {
    const latestState = sessionState;
    const {
      search,
      selectedAgentPolicies,
      selectedStatus,
      selectedTags,
      showUpgradeable
    } = defaultAgentListState;
    updateTableState({
      search,
      selectedAgentPolicies,
      selectedStatus,
      selectedTags,
      showUpgradeable,
      page: {
        ...latestState.page,
        index: 0 // Reset to first page
      }
    });
  }, [sessionState, updateTableState]);
  return {
    // Current state
    search: sessionState.search,
    selectedAgentPolicies: sessionState.selectedAgentPolicies,
    selectedStatus: sessionState.selectedStatus,
    selectedTags: sessionState.selectedTags,
    showUpgradeable: sessionState.showUpgradeable,
    sort: sessionState.sort,
    page: sessionState.page,
    // Utility functions
    clearFilters,
    updateTableState,
    onTableChange
  };
};
exports.useSessionAgentListState = useSessionAgentListState;