"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isAgentlessIntegration = exports.getAgentlessGlobalDataTags = exports.getAgentlessAgentPolicyNameFromPackagePolicyName = void 0;
exports.isInputAllowedForDeploymentMode = isInputAllowedForDeploymentMode;
exports.isOnlyAgentlessPolicyTemplate = exports.isOnlyAgentlessIntegration = void 0;
exports.validateDeploymentModesForInputs = validateDeploymentModesForInputs;
var _constants = require("../constants");
var _errors = require("../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Extract registry inputs from package info for deployment mode checking.
 * Keyed by both input type and policy template name.
 */
function extractRegistryInputsForDeploymentMode(packageInfo) {
  var _packageInfo$policy_t;
  const inputs = [];
  packageInfo === null || packageInfo === void 0 ? void 0 : (_packageInfo$policy_t = packageInfo.policy_templates) === null || _packageInfo$policy_t === void 0 ? void 0 : _packageInfo$policy_t.forEach(template => {
    if ('inputs' in template && template.inputs) {
      template.inputs.forEach(input => {
        if (!inputs.find(i => i.type === input.type && i.policy_template === template.name)) {
          inputs.push({
            type: input.type,
            policy_template: template.name,
            deployment_modes: input.deployment_modes
          });
        }
      });
    }
  });
  return inputs;
}

// Checks if a package has a policy template that supports agentless
// Provide a specific integration policy template name to check if it alone supports agentless
const isAgentlessIntegration = (packageInfo, integrationToEnable) => {
  var _packageInfo$policy_t6;
  if (integrationToEnable) {
    var _packageInfo$policy_t2, _packageInfo$policy_t3, _packageInfo$policy_t4, _packageInfo$policy_t5;
    return Boolean((packageInfo === null || packageInfo === void 0 ? void 0 : (_packageInfo$policy_t2 = packageInfo.policy_templates) === null || _packageInfo$policy_t2 === void 0 ? void 0 : (_packageInfo$policy_t3 = _packageInfo$policy_t2.find(({
      name
    }) => name === integrationToEnable)) === null || _packageInfo$policy_t3 === void 0 ? void 0 : (_packageInfo$policy_t4 = _packageInfo$policy_t3.deployment_modes) === null || _packageInfo$policy_t4 === void 0 ? void 0 : (_packageInfo$policy_t5 = _packageInfo$policy_t4.agentless) === null || _packageInfo$policy_t5 === void 0 ? void 0 : _packageInfo$policy_t5.enabled) === true);
  }
  return Boolean(packageInfo === null || packageInfo === void 0 ? void 0 : (_packageInfo$policy_t6 = packageInfo.policy_templates) === null || _packageInfo$policy_t6 === void 0 ? void 0 : _packageInfo$policy_t6.some(policyTemplate => {
    var _policyTemplate$deplo, _policyTemplate$deplo2;
    return (policyTemplate === null || policyTemplate === void 0 ? void 0 : (_policyTemplate$deplo = policyTemplate.deployment_modes) === null || _policyTemplate$deplo === void 0 ? void 0 : (_policyTemplate$deplo2 = _policyTemplate$deplo.agentless) === null || _policyTemplate$deplo2 === void 0 ? void 0 : _policyTemplate$deplo2.enabled) === true;
  }));
};
exports.isAgentlessIntegration = isAgentlessIntegration;
const getAgentlessAgentPolicyNameFromPackagePolicyName = packagePolicyName => {
  return `Agentless policy for ${packagePolicyName}`;
};
exports.getAgentlessAgentPolicyNameFromPackagePolicyName = getAgentlessAgentPolicyNameFromPackagePolicyName;
const isOnlyAgentlessIntegration = (packageInfo, integrationToEnable) => {
  var _packageInfo$policy_t7, _packageInfo$policy_t8, _packageInfo$policy_t9;
  if (packageInfo && packageInfo.policy_templates && ((_packageInfo$policy_t7 = packageInfo.policy_templates) === null || _packageInfo$policy_t7 === void 0 ? void 0 : _packageInfo$policy_t7.length) > 0 && (integrationToEnable && (_packageInfo$policy_t8 = packageInfo.policy_templates) !== null && _packageInfo$policy_t8 !== void 0 && _packageInfo$policy_t8.find(p => p.name === integrationToEnable && isOnlyAgentlessPolicyTemplate(p)) || (_packageInfo$policy_t9 = packageInfo.policy_templates) !== null && _packageInfo$policy_t9 !== void 0 && _packageInfo$policy_t9.every(p => isOnlyAgentlessPolicyTemplate(p)))) {
    return true;
  }
  return false;
};
exports.isOnlyAgentlessIntegration = isOnlyAgentlessIntegration;
const isOnlyAgentlessPolicyTemplate = policyTemplate => {
  var _policyTemplate$deplo3;
  return Boolean(policyTemplate.deployment_modes && ((_policyTemplate$deplo3 = policyTemplate.deployment_modes.agentless) === null || _policyTemplate$deplo3 === void 0 ? void 0 : _policyTemplate$deplo3.enabled) === true && (!policyTemplate.deployment_modes.default || policyTemplate.deployment_modes.default.enabled === false));
};

/*
 * Check if an input is allowed for a specific deployment mode based on its deployment_modes property.
 * If deployment_modes is not present, check against the input type blocklist instead.
 */
exports.isOnlyAgentlessPolicyTemplate = isOnlyAgentlessPolicyTemplate;
function isInputAllowedForDeploymentMode(input, deploymentMode, packageInfo) {
  // Always allow system package for monitoring, if this is not excluded it will be blocked
  // by the following code because it contains `logfile` input which is in the blocklist.
  if ((packageInfo === null || packageInfo === void 0 ? void 0 : packageInfo.name) === 'system') {
    return true;
  }

  // Check first if policy_template for input supports the deployment type
  if (packageInfo && input.policy_template && !integrationSupportsDeploymentMode(deploymentMode, packageInfo, input.policy_template)) {
    return false;
  }

  // Find the registry input definition for this input type and policy template
  const registryInput = extractRegistryInputsForDeploymentMode(packageInfo).find(rInput => rInput.type === input.type && (input.policy_template ? input.policy_template === rInput.policy_template : true));

  // If deployment_modes is specified in the registry, use it
  if (registryInput !== null && registryInput !== void 0 && registryInput.deployment_modes && Array.isArray(registryInput.deployment_modes)) {
    return registryInput.deployment_modes.includes(deploymentMode);
  }

  // For backward compatibility, if deployment_modes is not specified:
  // - For agentless mode, check the blocklist
  // - For default mode, allow all inputs
  if (deploymentMode === 'agentless') {
    return !_constants.AGENTLESS_DISABLED_INPUTS.includes(input.type);
  }
  return true; // Allow all inputs for default mode when deployment_modes is not specified
}
const integrationSupportsDeploymentMode = (deploymentMode, packageInfo, integrationName) => {
  var _packageInfo$policy_t10, _integration$deployme;
  if (deploymentMode === 'agentless') {
    return isAgentlessIntegration(packageInfo, integrationName);
  }
  const integration = (_packageInfo$policy_t10 = packageInfo.policy_templates) === null || _packageInfo$policy_t10 === void 0 ? void 0 : _packageInfo$policy_t10.find(({
    name
  }) => name === integrationName);
  if (integration !== null && integration !== void 0 && (_integration$deployme = integration.deployment_modes) !== null && _integration$deployme !== void 0 && _integration$deployme.default) {
    var _integration$deployme2;
    return (_integration$deployme2 = integration.deployment_modes) === null || _integration$deployme2 === void 0 ? void 0 : _integration$deployme2.default.enabled;
  }
  return true;
};

/*
 * Throw error if trying to enabling an input that is not allowed in agentless
 */
function validateDeploymentModesForInputs(inputs, deploymentMode, packageInfo) {
  inputs.forEach(input => {
    if (input.enabled && !isInputAllowedForDeploymentMode(input, deploymentMode, packageInfo)) {
      throw new _errors.PackagePolicyValidationError(`Input ${input.type}${packageInfo !== null && packageInfo !== void 0 && packageInfo.name ? ` in ${packageInfo.name}` : ''} is not allowed for deployment mode '${deploymentMode}'`);
    }
  });
}

/**
 * Derive global data tags for agentless agent policies from package agentless info.
 */
const getAgentlessGlobalDataTags = packageInfo => {
  var _packageInfo$policy_t11, _agentlessPolicyTempl;
  if (!(packageInfo !== null && packageInfo !== void 0 && packageInfo.policy_templates) && !(packageInfo !== null && packageInfo !== void 0 && (_packageInfo$policy_t11 = packageInfo.policy_templates) !== null && _packageInfo$policy_t11 !== void 0 && _packageInfo$policy_t11.some(policy => policy.deployment_modes))) {
    return undefined;
  }
  const agentlessPolicyTemplate = packageInfo.policy_templates.find(policy => policy.deployment_modes);

  // assumes that all the policy templates agentless deployments modes indentify have the same organization, division and team
  const agentlessInfo = agentlessPolicyTemplate === null || agentlessPolicyTemplate === void 0 ? void 0 : (_agentlessPolicyTempl = agentlessPolicyTemplate.deployment_modes) === null || _agentlessPolicyTempl === void 0 ? void 0 : _agentlessPolicyTempl.agentless;
  if (agentlessInfo === undefined || !agentlessInfo.organization || !agentlessInfo.division || !agentlessInfo.team) {
    return undefined;
  }
  return [{
    name: _constants.AGENTLESS_GLOBAL_TAG_NAME_ORGANIZATION,
    value: agentlessInfo.organization
  }, {
    name: _constants.AGENTLESS_GLOBAL_TAG_NAME_DIVISION,
    value: agentlessInfo.division
  }, {
    name: _constants.AGENTLESS_GLOBAL_TAG_NAME_TEAM,
    value: agentlessInfo.team
  }];
};
exports.getAgentlessGlobalDataTags = getAgentlessGlobalDataTags;