"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FieldsMetadataClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _fields_metadata_dictionary = require("../../../common/fields_metadata/models/fields_metadata_dictionary");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class FieldsMetadataClient {
  constructor(capabilities, logger, ecsFieldsRepository, metadataFieldsRepository, integrationFieldsRepository, otelFieldsRepository) {
    (0, _defineProperty2.default)(this, "makeSourceValidator", sourceList => source => {
      const sources = Array.isArray(sourceList) ? sourceList : [sourceList];
      // When `source` filter is not provided, all sources are allowed
      return (0, _lodash.isEmpty)(sources) ? true : sources.includes(source);
    });
    this.capabilities = capabilities;
    this.logger = logger;
    this.ecsFieldsRepository = ecsFieldsRepository;
    this.metadataFieldsRepository = metadataFieldsRepository;
    this.integrationFieldsRepository = integrationFieldsRepository;
    this.otelFieldsRepository = otelFieldsRepository;
  }
  async getByName(fieldName, {
    integration,
    dataset,
    source = []
  } = {}) {
    this.logger.debug(`Retrieving field metadata for: ${fieldName}`);
    const isSourceAllowed = this.makeSourceValidator(source);
    let field;

    // 1. Try resolving from metadata-fields static metadata (highest priority)
    if (isSourceAllowed('metadata')) {
      field = this.metadataFieldsRepository.getByName(fieldName);
    }

    // 2. Try resolving from ECS static metadata (authoritative schema)
    if (!field && isSourceAllowed('ecs')) {
      field = this.ecsFieldsRepository.getByName(fieldName);
    }

    // 3. Try resolving from OpenTelemetry semantic conventions (fallback)
    if (!field && isSourceAllowed('otel')) {
      field = this.otelFieldsRepository.getByName(fieldName);
    }

    // 4. Try searching for the field in the Elastic Package Registry (integration-specific)
    if (!field && isSourceAllowed('integration') && this.hasFleetPermissions(this.capabilities)) {
      field = await this.integrationFieldsRepository.getByName(fieldName, {
        integration,
        dataset
      });
    }
    return field;
  }
  async find({
    fieldNames,
    integration,
    dataset,
    source = []
  } = {}) {
    const isSourceAllowed = this.makeSourceValidator(source);
    if (!fieldNames) {
      /**
       * The merge order is important here.
       * The metadata fields repository has the highest priority,
       * followed by the ECS fields repository,
       * followed by the OpenTel fields repository.
       *
       * This is because we want the ECS fields repository to be more authoritative than the OpenTel fields repository.
       */
      return _fields_metadata_dictionary.FieldsMetadataDictionary.create({
        ...(isSourceAllowed('otel') && this.otelFieldsRepository.find().getFields()),
        ...(isSourceAllowed('ecs') && this.ecsFieldsRepository.find().getFields()),
        ...(isSourceAllowed('metadata') && this.metadataFieldsRepository.find().getFields())
      });
    }
    const fields = {};
    for (const fieldName of fieldNames) {
      const field = await this.getByName(fieldName, {
        integration,
        dataset,
        source
      });
      if (field) {
        fields[fieldName] = field;
      }
    }
    return _fields_metadata_dictionary.FieldsMetadataDictionary.create(fields);
  }
  hasFleetPermissions(capabilities) {
    const {
      fleet,
      fleetv2
    } = capabilities;
    return fleet.read && fleetv2.read;
  }
  static create({
    capabilities,
    logger,
    ecsFieldsRepository,
    metadataFieldsRepository,
    integrationFieldsRepository,
    otelFieldsRepository
  }) {
    return new FieldsMetadataClient(capabilities, logger, ecsFieldsRepository, metadataFieldsRepository, integrationFieldsRepository, otelFieldsRepository);
  }
}
exports.FieldsMetadataClient = FieldsMetadataClient;