"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.handleListOperator = handleListOperator;
exports.handleNullCheckOperator = handleNullCheckOperator;
exports.handleStringListOperator = handleStringListOperator;
var _is = require("../../../../../ast/is");
var _complete_items = require("../../../../../commands_registry/complete_items");
var _expressions = require("../../../expressions");
var _utils = require("./utils");
var _comma_decision_engine = require("../comma_decision_engine");
var _literals = require("../../../literals");
var _suggestion_builder = require("../suggestion_builder");
var _helpers = require("../../helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// ============================================================================
// eg. IN / NOT IN Operators
// ============================================================================

/** Handles IN and NOT IN operators with list syntax (e.g., field IN (1, 2, 3)) */
async function handleListOperator(ctx) {
  const {
    expressionRoot,
    innerText
  } = ctx;
  const fn = expressionRoot;
  const list = (0, _expressions.getBinaryExpressionOperand)(fn, 'right');
  const leftOperand = (0, _expressions.getBinaryExpressionOperand)(fn, 'left');

  // No list yet: suggest opening parenthesis
  if ((0, _utils.shouldSuggestOpenListForOperand)(list)) {
    return [_complete_items.listCompleteItem];
  }

  // List exists
  if ((0, _is.isList)(list)) {
    // Cursor after closed list: suggest logical operators
    if (innerText.length > list.location.max) {
      return (0, _utils.getLogicalContinuationSuggestions)();
    }

    // Cursor inside list - try to find the column to determine type
    let columnForType;
    if ((0, _is.isColumn)(leftOperand)) {
      columnForType = leftOperand;
    } else {
      // Try to extract the actual field from fn.args[0]
      const firstArg = fn.args[0];
      if ((0, _is.isColumn)(firstArg)) {
        columnForType = firstArg;
      }
    }

    // After a value but not after comma: suggest comma
    if ((0, _comma_decision_engine.shouldSuggestComma)({
      position: 'inside_list',
      innerText,
      listHasValues: list.values && list.values.length > 0
    })) {
      return [(0, _helpers.withAutoSuggest)({
        ..._complete_items.commaCompleteItem,
        text: ','
      })];
    }

    // After comma or empty list: suggest values
    const isEmptyList = !list.values || list.values.length === 0;
    const ignoredColumns = isEmptyList ? [] : [...(columnForType ? [columnForType.parts.join('.')] : []), ...(list.values || []).filter(_is.isColumn).map(col => col.parts.join('.'))].filter(Boolean);
    return getListValueSuggestions(ctx, columnForType, ignoredColumns);
  }
  return [];
}
async function getListValueSuggestions(ctx, column, ignoredColumns) {
  var _ctx$context;
  const argType = column ? (0, _expressions.getExpressionType)(column, (_ctx$context = ctx.context) === null || _ctx$context === void 0 ? void 0 : _ctx$context.columns) : undefined;
  const builder = new _suggestion_builder.SuggestionBuilder(ctx);
  await builder.addFields({
    types: argType ? [argType] : undefined,
    ignoredColumns,
    addSpaceAfterField: true,
    openSuggestions: true
  });
  builder.addFunctions({
    types: argType ? [argType] : undefined
  }).addLiterals({
    types: argType ? [argType] : undefined,
    includeDateLiterals: true,
    includeCompatibleLiterals: true
  });
  return builder.build();
}

// ============================================================================
// eg. IS NULL / IS NOT NULL Operators
// ============================================================================

/** Handles NULL-check operators (IS NULL, IS NOT NULL) */
async function handleNullCheckOperator(ctx) {
  const fn = ctx.expressionRoot;
  if ((fn === null || fn === void 0 ? void 0 : fn.incomplete) === true) {
    return null;
  }
  return (0, _utils.getLogicalContinuationSuggestions)();
}

// ============================================================================
// eg. LIKE / RLIKE / NOT LIKE / NOT RLIKE Operators
// ============================================================================

/** Handles string pattern operators with list syntax (e.g., field LIKE ("*pattern*", "*other*")) */
async function handleStringListOperator(context) {
  const fn = context.expressionRoot;
  if (!fn) {
    return null;
  }
  const operator = fn.name.toLowerCase();
  const rightOperand = (0, _expressions.getBinaryExpressionOperand)(fn, 'right');
  const leftOperand = (0, _expressions.getBinaryExpressionOperand)(fn, 'left');

  // No list yet: suggest any string expressions (LIKE pattern can be any string expression)
  if ((0, _utils.shouldSuggestOpenListForOperand)(rightOperand)) {
    // LIKE/RLIKE accepts any string pattern, so suggest all string-compatible expressions
    const builder = new _suggestion_builder.SuggestionBuilder(context);
    const ignoredColumns = (0, _is.isColumn)(leftOperand) ? [leftOperand.parts.join('.')].filter(Boolean) : [];
    await builder.addFields({
      types: ['any'],
      ignoredColumns
    });
    builder.addFunctions({
      types: ['any']
    }).addLiterals({
      types: ['any'],
      includeDateLiterals: false,
      includeCompatibleLiterals: true
    });
    return builder.build();
  }

  // Only handle list form; otherwise, delegate by returning null
  if (!(0, _is.isList)(rightOperand)) {
    return null;
  }
  const list = rightOperand;
  const cursorPos = context.innerText.length;

  // Cursor after closed list: suggest logical operators
  if (cursorPos > list.location.max) {
    return (0, _utils.getLogicalContinuationSuggestions)();
  }

  // Cursor inside list: if previous value without a comma, suggest comma
  if ((0, _comma_decision_engine.shouldSuggestComma)({
    position: 'inside_list',
    innerText: context.innerText,
    listHasValues: list.values && list.values.length > 0
  })) {
    return [{
      ..._complete_items.commaCompleteItem,
      text: ', '
    }];
  }

  // Otherwise suggest some basic patterns based on operator
  return getStringPatternSuggestions(operator);
}

/** Returns basic pattern suggestions for LIKE/RLIKE operators */
function getStringPatternSuggestions(operator) {
  const isRlike = operator.includes('rlike');

  // RLIKE: empty string, match anything, match from start to end
  // LIKE: empty string, wildcard for any characters
  const patterns = isRlike ? ['""', '".*"', '"^.*$"'] : ['""', '"*"'];
  return (0, _literals.buildConstantsDefinitions)(patterns, undefined, 'A');
}