"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getMissingIndexPrivilegesInternalRoute = exports.getMissingIndexPrivileges = void 0;
var _zodHelpers = require("@kbn/zod-helpers");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _actions = require("@kbn/security-solution-features/actions");
var _get_scheduled_index_pattern = require("../../../lib/attack_discovery/persistence/get_scheduled_index_pattern");
var _build_response = require("../../../lib/build_response");
var _helpers = require("../../helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const REQUIRED_INDEX_PRIVILEGES = ['read', 'write', 'view_index_metadata', 'maintenance'];
const getMissingIndexPrivileges = (indexName, privilegesResponse) => {
  var _privilegesResponse$i;
  const privileges = (_privilegesResponse$i = privilegesResponse.index[indexName]) !== null && _privilegesResponse$i !== void 0 ? _privilegesResponse$i : {};
  const missingPrivileges = REQUIRED_INDEX_PRIVILEGES.filter(privilege => !privileges[privilege]);
  if (missingPrivileges.length) {
    return {
      index_name: indexName,
      privileges: missingPrivileges
    };
  }
};
exports.getMissingIndexPrivileges = getMissingIndexPrivileges;
const getMissingIndexPrivilegesInternalRoute = router => {
  router.versioned.get({
    access: 'internal',
    path: _elasticAssistantCommon.ATTACK_DISCOVERY_INTERNAL_MISSING_PRIVILEGES,
    security: {
      authz: {
        requiredPrivileges: [_actions.ATTACK_DISCOVERY_API_ACTION_ALL]
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.internal.v1,
    validate: {
      response: {
        200: {
          body: {
            custom: (0, _zodHelpers.buildRouteValidationWithZod)(_elasticAssistantCommon.GetAttackDiscoveryMissingPrivilegesInternalResponse)
          }
        }
      }
    }
  }, async (context, request, response) => {
    const ctx = await context.resolve(['core', 'elasticAssistant', 'licensing']);
    const resp = (0, _build_response.buildResponse)(response);
    const assistantContext = await context.elasticAssistant;
    const logger = assistantContext.logger;

    // Perform license and authenticated user
    const checkResponse = await (0, _helpers.performChecks)({
      context: ctx,
      request,
      response
    });
    if (!checkResponse.isSuccess) {
      return checkResponse.response;
    }
    try {
      const dataClient = await assistantContext.getAttackDiscoveryDataClient();
      if (!dataClient) {
        return resp.error({
          body: `Attack discovery data client not initialized`,
          statusCode: 500
        });
      }
      const core = await context.core;
      const esClient = core.elasticsearch.client.asCurrentUser;
      const spaceId = dataClient.spaceId;
      const indexPattern = (0, _get_scheduled_index_pattern.getScheduledIndexPattern)(spaceId);
      const adhocIndexPattern = dataClient.getAdHocAlertsIndexPattern();
      const privileges = await readIndexPrivileges(esClient, [indexPattern, adhocIndexPattern]);
      const missingPrivileges = [];
      const missingIndexPrivileges = getMissingIndexPrivileges(indexPattern, privileges);
      if (missingIndexPrivileges) {
        missingPrivileges.push(missingIndexPrivileges);
      }
      const missingAdhocIndexPrivileges = getMissingIndexPrivileges(adhocIndexPattern, privileges);
      if (missingAdhocIndexPrivileges) {
        missingPrivileges.push(missingAdhocIndexPrivileges);
      }
      return response.ok({
        body: missingPrivileges
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return resp.error({
        body: {
          success: false,
          error: error.message
        },
        statusCode: error.statusCode
      });
    }
  });
};
exports.getMissingIndexPrivilegesInternalRoute = getMissingIndexPrivilegesInternalRoute;
const readIndexPrivileges = async (esClient, indices) => {
  const response = await esClient.security.hasPrivileges({
    index: [{
      names: indices,
      privileges: REQUIRED_INDEX_PRIVILEGES
    }]
  }, {
    meta: true
  });
  return response.body;
};