"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDissectProcessorWithReview = getDissectProcessorWithReview;
var _collapse_repeats = require("./collapse_repeats");
var _serialize_ast = require("../serialize_ast");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Generates a Dissect processor by combining extracted pattern and
 * the result of an LLM review. It replaces generic field names (field_1, field_2, etc.)
 * with ECS-compliant field names and handles field grouping.
 */
function getDissectProcessorWithReview(pattern, reviewResult, sourceField = 'message') {
  var _pattern$fields$0$val, _pattern$fields$;
  // Build a mapping from original field names to ECS field names
  const fieldNameMap = new Map();
  const fieldGroupMap = new Map();
  reviewResult.fields.forEach(field => {
    field.columns.forEach(columnName => {
      fieldNameMap.set(columnName, field.ecs_field);
    });
    if (field.columns.length > 1) {
      fieldGroupMap.set(field.ecs_field, field.columns);
    }
  });

  // Transform the AST with ECS field names and handle grouping
  const transformedAST = transformASTWithReview(pattern.ast, fieldNameMap, fieldGroupMap);

  // Collapse repeated field sequences
  const collapsedAST = (0, _collapse_repeats.collapseRepeats)(transformedAST);

  // Serialize to pattern string
  const finalPattern = (0, _serialize_ast.serializeAST)(collapsedAST);

  // Build new fields list
  const newFields = [];
  const processedGroups = new Set();
  pattern.fields.forEach(field => {
    const ecsFieldName = fieldNameMap.get(field.name);
    if (!ecsFieldName) {
      // Field not mapped by LLM, keep original
      newFields.push(field);
      return;
    }
    const fieldGroup = fieldGroupMap.get(ecsFieldName);
    if (fieldGroup && fieldGroup.length > 1) {
      // This field is part of a multi-column group
      if (!processedGroups.has(ecsFieldName)) {
        // First field in the group - create the combined field
        const groupFields = pattern.fields.filter(f => fieldGroup.includes(f.name));
        const combinedValues = groupFields.map(f => f.values);

        // Combine values from all fields in the group
        const mergedValues = [];
        for (let i = 0; i < Math.max(...combinedValues.map(v => v.length)); i++) {
          const parts = [];
          combinedValues.forEach(values => {
            if (values[i] !== undefined) {
              parts.push(values[i]);
            }
          });
          if (parts.length > 0) {
            mergedValues.push(parts.join(' '));
          }
        }
        newFields.push({
          name: ecsFieldName,
          values: mergedValues,
          position: field.position,
          modifiers: field.modifiers
        });
        processedGroups.add(ecsFieldName);
      }
    } else {
      // Single-column field, just rename it
      newFields.push({
        ...field,
        name: ecsFieldName
      });
    }
  });
  const dissectConfig = {
    field: sourceField,
    pattern: finalPattern,
    ignore_missing: true
  };

  // Add append_separator since all multi-column fields use append strategy
  const hasMultiColumnFields = reviewResult.fields.some(field => field.columns.length > 1);
  if (hasMultiColumnFields) {
    dissectConfig.append_separator = ' ';
  }
  return {
    description: reviewResult.log_source,
    pattern: finalPattern,
    processor: {
      dissect: dissectConfig
    },
    metadata: {
      messageCount: (_pattern$fields$0$val = (_pattern$fields$ = pattern.fields[0]) === null || _pattern$fields$ === void 0 ? void 0 : _pattern$fields$.values.length) !== null && _pattern$fields$0$val !== void 0 ? _pattern$fields$0$val : 0,
      fieldCount: newFields.length
    }
  };
}

/**
 * Transform AST nodes based on review result
 * Replaces field names with ECS names and adds appropriate modifiers for grouped fields
 */
function transformASTWithReview(ast, fieldNameMap, fieldGroupMap) {
  const transformedNodes = [];
  for (let i = 0; i < ast.nodes.length; i++) {
    const node = ast.nodes[i];
    if (node.type === 'literal') {
      transformedNodes.push(node);
      continue;
    }

    // Field node
    const fieldNode = node;
    const originalName = fieldNode.name;
    const ecsFieldName = fieldNameMap.get(originalName);
    if (!ecsFieldName) {
      // Field not mapped by LLM, convert to skip field %{?}
      transformedNodes.push({
        type: 'field',
        name: '',
        modifiers: {
          skip: true,
          namedSkip: true
        }
      });
      continue;
    }
    const fieldGroup = fieldGroupMap.get(ecsFieldName);
    if (fieldGroup && fieldGroup.length > 1) {
      // This is part of a multi-column group - all fields use append modifier
      transformedNodes.push({
        type: 'field',
        name: ecsFieldName,
        modifiers: {
          ...fieldNode.modifiers,
          append: true
        }
      });
    } else {
      // Single-column field, just rename it
      transformedNodes.push({
        type: 'field',
        name: ecsFieldName,
        modifiers: fieldNode.modifiers
      });
    }
  }
  return {
    nodes: transformedNodes
  };
}