"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSharingData = getSharingData;
exports.showPublicUrlSwitch = void 0;
var _discoverUtils = require("@kbn/discover-utils");
var _state_comparators = require("../application/main/state_management/utils/state_comparators");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Preparing data to share the current state as link or CSV/Report
 */
async function getSharingData(currentSearchSource, state, services, isEsqlMode) {
  const {
    uiSettings,
    data
  } = services;
  const searchSource = currentSearchSource.createCopy();
  const index = searchSource.getField('index');
  searchSource.setField('sort', (0, _discoverUtils.getSortForSearchSource)({
    sort: state.sort,
    dataView: index,
    defaultSortDir: uiSettings.get(_discoverUtils.SORT_DEFAULT_ORDER_SETTING)
  }));
  searchSource.removeField('highlight');
  searchSource.removeField('highlightAll');
  searchSource.removeField('aggs');
  searchSource.removeField('size');

  // Columns that the user has selected in the saved search
  let columns = state.columns || [];
  if (columns && columns.length > 0) {
    // conditionally add the time field column:
    let timeFieldName;
    const hideTimeColumn = uiSettings.get(_discoverUtils.DOC_HIDE_TIME_COLUMN_SETTING);
    if (!hideTimeColumn && index && index.timeFieldName && !isEsqlMode) {
      timeFieldName = index.timeFieldName;
    }
    if (timeFieldName && !columns.includes(timeFieldName)) {
      columns = [timeFieldName, ...columns];
    }
  }
  const absoluteTimeFilter = data.query.timefilter.timefilter.createFilter(index);
  const relativeTimeFilter = data.query.timefilter.timefilter.createRelativeFilter(index);
  return {
    getSearchSource: ({
      addGlobalTimeFilter,
      absoluteTime
    }) => {
      let existingFilter = searchSource.getField('filter');
      const searchSourceUpdated = searchSource.createCopy();
      searchSourceUpdated.removeField('filter');
      const timeFilter = absoluteTime ? absoluteTimeFilter : relativeTimeFilter;
      if (addGlobalTimeFilter && timeFilter) {
        // remove timeFilter from existing filter
        if (Array.isArray(existingFilter)) {
          existingFilter = existingFilter.filter(current => !(0, _state_comparators.isEqualFilters)(current, absoluteTimeFilter));
        } else if ((0, _state_comparators.isEqualFilters)(existingFilter, absoluteTimeFilter)) {
          existingFilter = undefined;
        }
        if (existingFilter) {
          existingFilter = Array.isArray(existingFilter) ? [timeFilter, ...existingFilter] : [timeFilter, existingFilter];
        } else {
          existingFilter = timeFilter;
        }
      }
      if (existingFilter) {
        searchSourceUpdated.setField('filter', existingFilter);
      }

      /*
       * For downstream querying performance, the searchSource object must have fields set.
       * Otherwise, the requests will ask for all fields, even if only a few are really needed.
       * Discover does not set fields, since having all fields is needed for the UI.
       */
      searchSourceUpdated.removeField('fieldsFromSource');
      const fields = columns.length ? columns.map(column => {
        let field = column;

        // If this column is a nested field, add a wildcard to the field name in order to fetch
        // all leaf fields for the report, since the fields API doesn't support nested field roots
        if ((0, _discoverUtils.isNestedFieldParent)(column, index)) {
          field = `${column}.*`;
        }
        return {
          field,
          include_unmapped: true
        };
      }) : [{
        field: '*',
        include_unmapped: true
      }];
      searchSourceUpdated.setField('fields', fields);
      return searchSourceUpdated.getSerializedFields(true);
    },
    columns
  };
}
const showPublicUrlSwitch = anonymousUserCapabilities => {
  if (!anonymousUserCapabilities.discover_v2) return false;
  const discover = anonymousUserCapabilities.discover_v2;
  return !!discover.show;
};
exports.showPublicUrlSwitch = showPublicUrlSwitch;