"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchEsql = fetchEsql;
exports.getTextBasedQueryStateToAstProps = getTextBasedQueryStateToAstProps;
var _rxjs = require("rxjs");
var _i18n = require("@kbn/i18n");
var _common = require("@kbn/data-plugin/common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function fetchEsql({
  query,
  inputQuery,
  filters,
  timeRange,
  dataView,
  abortSignal,
  inspectorAdapters,
  data,
  expressions,
  scopedProfilesManager,
  esqlVariables,
  searchSessionId,
  projectRouting
}) {
  const props = getTextBasedQueryStateToAstProps({
    query,
    inputQuery,
    filters,
    timeRange,
    dataView,
    data
  });
  return (0, _common.textBasedQueryStateToAstWithValidation)(props).then(ast => {
    if (ast) {
      const contract = expressions.execute(ast, null, {
        inspectorAdapters,
        searchContext: {
          timeRange,
          esqlVariables,
          projectRouting
        },
        searchSessionId
      });
      abortSignal === null || abortSignal === void 0 ? void 0 : abortSignal.addEventListener('abort', contract.cancel);
      const execution = contract.getData();
      let finalData = [];
      let esqlQueryColumns;
      let error;
      let esqlHeaderWarning;
      execution.pipe((0, _rxjs.pluck)('result')).subscribe(resp => {
        const response = resp;
        if (response.type === 'error') {
          error = response.error.message;
        } else {
          var _table$rows, _table$columns, _table$warning;
          const table = response;
          const rows = (_table$rows = table === null || table === void 0 ? void 0 : table.rows) !== null && _table$rows !== void 0 ? _table$rows : [];
          esqlQueryColumns = (_table$columns = table === null || table === void 0 ? void 0 : table.columns) !== null && _table$columns !== void 0 ? _table$columns : undefined;
          esqlHeaderWarning = (_table$warning = table.warning) !== null && _table$warning !== void 0 ? _table$warning : undefined;
          finalData = rows.map((row, idx) => {
            const record = {
              id: String(idx),
              raw: row,
              flattened: row
            };
            return scopedProfilesManager.resolveDocumentProfile({
              record
            });
          });
        }
      });
      return (0, _rxjs.lastValueFrom)(execution).then(() => {
        if (error) {
          throw new Error(error);
        } else {
          const adapter = inspectorAdapters.requests;
          const interceptedWarnings = [];
          if (adapter) {
            data.search.showWarnings(adapter, warning => {
              interceptedWarnings.push(warning);
              return true; // suppress the default behaviour
            });
          }
          return {
            records: finalData || [],
            interceptedWarnings,
            esqlQueryColumns,
            esqlHeaderWarning
          };
        }
      });
    }
    return {
      records: [],
      interceptedWarnings: [],
      esqlQueryColumns: [],
      esqlHeaderWarning: undefined
    };
  }).catch(err => {
    throw new Error(err.message);
  });
}
function getTextBasedQueryStateToAstProps({
  query,
  inputQuery,
  filters,
  timeRange,
  dataView,
  data
}) {
  return {
    filters,
    query,
    time: timeRange !== null && timeRange !== void 0 ? timeRange : data.query.timefilter.timefilter.getAbsoluteTime(),
    timeFieldName: dataView.timeFieldName,
    inputQuery,
    titleForInspector: _i18n.i18n.translate('discover.inspectorEsqlRequestTitle', {
      defaultMessage: 'Table'
    }),
    descriptionForInspector: _i18n.i18n.translate('discover.inspectorEsqlRequestDescription', {
      defaultMessage: 'This request queries Elasticsearch to fetch results for the table.'
    })
  };
}