"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initializeIndexTemplate = initializeIndexTemplate;
var _nodeAssert = _interopRequireDefault(require("node:assert"));
var _retry_es = require("../retry_es");
var _defaults = require("./defaults");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * https://www.elastic.co/docs/manage-data/data-store/data-streams/set-up-data-stream
 *
 * Endeavour to be idempotent and race-condition safe.
 */
async function initializeIndexTemplate({
  logger,
  dataStream,
  elasticsearchClient,
  existingIndexTemplate,
  skipCreation = true
}) {
  const version = dataStream.version;
  logger.debug(`Setting up index template for data stream: ${dataStream.name} v${version}`);
  const previousVersions = [];
  dataStream = (0, _defaults.applyDefaults)(dataStream);
  if (skipCreation && !existingIndexTemplate) {
    // index template does not exist so it is not updated.
    logger.debug(`Skipping index template creation during lazy initialization: ${dataStream.name}.`);
    return {
      uptoDate: false
    };
  }

  // index template exists so we always update it.
  if (existingIndexTemplate) {
    var _existingIndexTemplat, _existingIndexTemplat2, _existingIndexTemplat3, _existingIndexTemplat4;
    logger.debug(`Index template already exists: ${dataStream.name}, updating it.`);
    const deployedVersion = (_existingIndexTemplat = existingIndexTemplate.index_template) === null || _existingIndexTemplat === void 0 ? void 0 : (_existingIndexTemplat2 = _existingIndexTemplat._meta) === null || _existingIndexTemplat2 === void 0 ? void 0 : _existingIndexTemplat2.version;
    (0, _nodeAssert.default)(typeof deployedVersion === 'number' && deployedVersion > 0, `Datastream ${dataStream.name} metadata is in an unexpected state, expected version to be a number but got ${deployedVersion}`);
    if (deployedVersion >= version) {
      // index already applied and updated.
      logger.debug(`Deployed ${dataStream.name} v${deployedVersion} already applied and updated.`);
      return {
        uptoDate: true
      };
    }
    previousVersions.push(deployedVersion, ...((_existingIndexTemplat3 = existingIndexTemplate.index_template) === null || _existingIndexTemplat3 === void 0 ? void 0 : (_existingIndexTemplat4 = _existingIndexTemplat3._meta) === null || _existingIndexTemplat4 === void 0 ? void 0 : _existingIndexTemplat4.previousVersions));
  }
  logger.debug(`Putting index template: ${dataStream.name}.`);
  // Should be idempotent
  await (0, _retry_es.retryEs)(() => elasticsearchClient.indices.putIndexTemplate({
    name: dataStream.name,
    priority: dataStream.template.priority,
    index_patterns: [`${dataStream.name}*`],
    composed_of: dataStream.template.composedOf,
    data_stream: {
      hidden: dataStream.hidden
    },
    template: {
      aliases: dataStream.template.aliases,
      mappings: dataStream.template.mappings,
      settings: dataStream.template.settings
    },
    _meta: {
      ...dataStream.template._meta,
      version,
      previousVersions
    }
  }));

  // index template updated
  return {
    uptoDate: true
  };
}