"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.esqlAsyncSearchStrategyProvider = void 0;
var _rxjs = require("rxjs");
var _server = require("@kbn/kibana-utils-plugin/server");
var _response_utils = require("./response_utils");
var _async_utils = require("../common/async_utils");
var _common = require("../../../../common");
var _report_search_error = require("../../report_search_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// `drop_null_columns` is going to change the response
// now we get `all_columns` and `columns`
// `columns` contain only columns with data
// `all_columns` contain everything

const esqlAsyncSearchStrategyProvider = (searchConfig, logger) => {
  function cancelEsqlAsyncSearch(id, {
    esClient
  }) {
    return esClient.asCurrentUser.transport.request({
      method: 'DELETE',
      path: `/_query/async/${id}`
    }, {
      meta: true,
      // we don't want the ES client to retry (default value is 3)
      maxRetries: 0
    });
  }
  function stopEsqlAsyncSearch(id, options, {
    esClient
  }) {
    return esClient.asCurrentUser.transport.request({
      method: 'POST',
      path: `/_query/async/${id}/stop`
    }, {
      ...options.transport,
      signal: options.abortSignal,
      meta: true,
      asStream: options.stream
    });
  }
  function getEsqlAsyncSearch({
    id,
    ...request
  }, options, {
    esClient
  }) {
    var _request$params, _request$params2, _request$params3;
    const params = {
      ...(0, _async_utils.getCommonDefaultAsyncGetParams)(searchConfig, options),
      ...((_request$params = request.params) !== null && _request$params !== void 0 && _request$params.keep_alive ? {
        keep_alive: request.params.keep_alive
      } : {}),
      ...((_request$params2 = request.params) !== null && _request$params2 !== void 0 && _request$params2.wait_for_completion_timeout ? {
        wait_for_completion_timeout: request.params.wait_for_completion_timeout
      } : {})
    };
    return esClient.asCurrentUser.transport.request({
      method: 'GET',
      path: `/_query/async/${id}`,
      // FIXME: the drop_null_columns param shouldn't be needed here once https://github.com/elastic/elasticsearch/issues/138439 is resolved
      querystring: {
        ...params,
        drop_null_columns: (_request$params3 = request.params) === null || _request$params3 === void 0 ? void 0 : _request$params3.dropNullColumns
      }
    }, {
      ...options.transport,
      signal: options.abortSignal,
      meta: true,
      asStream: options.stream
    });
  }
  async function submitEsqlSearch({
    id,
    ...request
  }, options, {
    esClient
  }) {
    var _request$params4;
    const {
      dropNullColumns,
      ...requestParams
    } = (_request$params4 = request.params) !== null && _request$params4 !== void 0 ? _request$params4 : {};
    const params = {
      ...(await (0, _async_utils.getCommonDefaultAsyncSubmitParams)(searchConfig, options)),
      ...requestParams
    };
    return esClient.asCurrentUser.transport.request({
      method: 'POST',
      path: `/_query/async`,
      body: params,
      querystring: dropNullColumns ? 'drop_null_columns' : ''
    }, {
      ...options.transport,
      signal: options.abortSignal,
      meta: true,
      asStream: options.stream
    });
  }
  function esqlAsyncSearch({
    id,
    ...request
  }, searchOptions, deps) {
    // This abortSignal comes from getRequestAbortedSignal and fires if the HTTP request is aborted;
    // in the case of these async APIs, we  don't want to cancel the async request if the HTTP
    // request is aborted
    const {
      abortSignal,
      ...options
    } = searchOptions;
    const search = async () => {
      var _meta$request;
      const response = await (!id ? submitEsqlSearch({
        id,
        ...request
      }, options, deps) : options.retrieveResults ? stopEsqlAsyncSearch(id, options, deps) : getEsqlAsyncSearch({
        id,
        ...request
      }, options, deps));
      const {
        body,
        headers,
        meta
      } = response;
      return (0, _response_utils.toAsyncKibanaSearchResponse)(body, headers, meta === null || meta === void 0 ? void 0 : (_meta$request = meta.request) === null || _meta$request === void 0 ? void 0 : _meta$request.params);
    };
    const cancel = async () => {
      if (!id || options.isStored) return;
      try {
        await cancelEsqlAsyncSearch(id, deps);
      } catch (e) {
        var _e$meta;
        // A 404 means either this search request does not exist, or that it is already cancelled
        if (((_e$meta = e.meta) === null || _e$meta === void 0 ? void 0 : _e$meta.statusCode) === 404) return;

        // Log all other (unexpected) error messages
        logger.error(`cancelEsqlAsyncSearch error: ${e.message}`);
      }
    };
    return (0, _common.pollSearch)(search, cancel, {
      pollInterval: searchConfig.asyncSearch.pollInterval,
      ...options
    }).pipe((0, _rxjs.tap)(response => id = response.id), (0, _rxjs.catchError)(e => {
      throw (0, _report_search_error.getKbnSearchError)(e);
    }));
  }
  return {
    /**
     * @param request
     * @param options
     * @param deps `SearchStrategyDependencies`
     * @returns `Observable<IKibanaResponse<SqlGetAsyncResponse>>`
     * @throws `KbnSearchError`
     */
    search: (request, options, deps) => {
      logger.debug(() => `search ${JSON.stringify(request) || request.id}`);
      return esqlAsyncSearch(request, options, deps);
    },
    /**
     * @param id async search ID to cancel, as returned from _async_search API
     * @param options
     * @param deps `SearchStrategyDependencies`
     * @returns `Promise<void>`
     * @throws `KbnServerError`
     */

    cancel: async (id, options, deps) => {
      logger.debug(`cancel ${id}`);
      try {
        await cancelEsqlAsyncSearch(id, deps);
      } catch (e) {
        throw (0, _server.getKbnServerError)(e);
      }
    },
    /**
     *
     * @param id async search ID to extend, as returned from _async_search API
     * @param keepAlive
     * @param options
     * @param deps `SearchStrategyDependencies`
     * @returns `Promise<void>`
     * @throws `KbnServerError`
     */
    extend: async (id, keepAlive, options, {
      esClient
    }) => {
      logger.debug(`extend ${id} by ${keepAlive}`);
      try {
        await esClient.asCurrentUser.transport.request({
          method: 'GET',
          path: `/_query/async/${id}`,
          querystring: {
            id,
            keep_alive: keepAlive
          }
        }, {
          ...options.transport,
          signal: options.abortSignal,
          meta: true
        });
      } catch (e) {
        throw (0, _server.getKbnServerError)(e);
      }
    }
  };
};
exports.esqlAsyncSearchStrategyProvider = esqlAsyncSearchStrategyProvider;