"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.dashboardClient = void 0;
var _lruCache = require("lru-cache");
var _public = require("@kbn/kibana-utils-plugin/public");
var _constants = require("../../common/constants");
var _kibana_services = require("../services/kibana_services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const CACHE_SIZE = 20; // only store a max of 20 dashboards
const CACHE_TTL = 1000 * 60 * 5; // time to live = 5 minutes

const cache = new _lruCache.LRUCache({
  max: CACHE_SIZE,
  ttl: CACHE_TTL
});
const dashboardClient = exports.dashboardClient = {
  create: async (dashboardState, references) => {
    return _kibana_services.coreServices.http.post(`/api/dashboards/dashboard`, {
      version: _constants.DASHBOARD_API_VERSION,
      query: {
        allowUnmappedKeys: true
      },
      body: JSON.stringify({
        data: {
          ...dashboardState,
          references
        }
      })
    });
  },
  delete: async id => {
    cache.delete(id);
    return _kibana_services.coreServices.http.delete(`/api/dashboards/dashboard/${id}`, {
      version: _constants.DASHBOARD_API_VERSION
    });
  },
  get: async id => {
    if (cache.has(id)) {
      return cache.get(id);
    }
    const result = await _kibana_services.coreServices.http.get(`/api/dashboards/dashboard/${id}`, {
      version: _constants.DASHBOARD_API_VERSION,
      query: {
        allowUnmappedKeys: true
      }
    }).catch(e => {
      var _e$response, _message, _e$body;
      if (((_e$response = e.response) === null || _e$response === void 0 ? void 0 : _e$response.status) === 404) {
        throw new _public.SavedObjectNotFound({
          type: _constants.DASHBOARD_SAVED_OBJECT_TYPE,
          id
        });
      }
      const message = (_message = (_e$body = e.body) === null || _e$body === void 0 ? void 0 : _e$body.message) !== null && _message !== void 0 ? _message : e.message;
      throw new Error(message);
    });
    if (result.meta.outcome !== 'aliasMatch') {
      /**
       * Only add the dashboard to the cache if it does not require a redirect - otherwise, the meta
       * alias info gets cached and prevents the dashboard contents from being updated
       */
      cache.set(id, result);
    }
    return result;
  },
  search: async searchBody => {
    return await _kibana_services.coreServices.http.post(`/api/dashboards/search`, {
      version: _constants.DASHBOARD_API_VERSION,
      body: JSON.stringify({
        ...searchBody,
        search: searchBody.search ? `${searchBody.search}*` : undefined
      })
    });
  },
  update: async (id, dashboardState, references) => {
    const updateResponse = await _kibana_services.coreServices.http.put(`/api/dashboards/dashboard/${id}`, {
      version: _constants.DASHBOARD_API_VERSION,
      query: {
        allowUnmappedKeys: true
      },
      body: JSON.stringify({
        data: {
          ...dashboardState,
          references
        }
      })
    });
    cache.delete(id);
    return updateResponse;
  }
};