"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.stripQueryParams = exports.parseNavigationTree = exports.flattenNav = exports.findActiveNodes = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const wrapIdx = index => `[${index}]`;

/**
 * Flatten the navigation tree into a record of path => node
 * for quicker access when detecting the active path
 *
 * @param navTree The navigation tree to flatten
 * @param prefix Array of path prefix (used in the recursion)
 * @returns The flattened navigation tree
 */
const flattenNav = (navTree, prefix = []) => {
  return navTree.reduce((acc, node, idx) => {
    const updatedPrefix = [...prefix, `${wrapIdx(idx)}`];
    const nodePath = () => updatedPrefix.join('');
    if (node.children && node.children.length > 0) {
      const {
        children,
        ...rest
      } = node;
      return {
        ...acc,
        [nodePath()]: rest,
        ...flattenNav(children, updatedPrefix)
      };
    }
    acc[nodePath()] = node;
    return acc;
  }, {});
};
exports.flattenNav = flattenNav;
function trim(str) {
  return divider => {
    const position = str.indexOf(divider);
    if (position !== -1) {
      str = str.slice(0, position);
    }
    return str;
  };
}
const stripQueryParams = url => trim(url)('?');
exports.stripQueryParams = stripQueryParams;
function serializeDeeplinkUrl(url) {
  if (!url) {
    return undefined;
  }
  return stripQueryParams(url);
}

/**
 * Extract the parent paths from a key
 *
 * @example
 * IN: "[0][1][2][0]"
 * OUT: ["[0]", "[0][1]", "[0][1][2]", "[0][1][2][0]"]
 *
 * @param key The key to extract parent paths from
 * @returns An array of parent paths
 */
function extractParentPaths(key, navTree) {
  // Split the string on every '][' to get an array of values without the brackets.
  const arr = key.split('][');
  if (arr.length === 1) {
    return arr;
  }
  // Add the brackets back in for the first and last elements, and all elements in between.
  arr[0] = `${arr[0]}]`;
  arr[arr.length - 1] = `[${arr[arr.length - 1]}`;
  for (let i = 1; i < arr.length - 1; i++) {
    arr[i] = `[${arr[i]}]`;
  }
  return arr.reduce((acc, _currentValue, currentIndex) => {
    acc.push(arr.slice(0, currentIndex + 1).join(''));
    return acc;
  }, []).filter(k => Boolean(navTree[k]));
}

/**
 * Find the active nodes in the navigation tree based on the current Location.pathname
 * Note that the pathname cand match multiple navigation tree branches, each branch
 * will be returned as an array of nodes.
 *
 * @param currentPathname The current Location.pathname
 * @param navTree The flattened navigation tree
 * @returns The active nodes
 */
const findActiveNodes = (currentPathname, navTree, location, prepend = path => path) => {
  const activeNodes = [];
  const matches = [];
  const activeNodeFromKey = key => ({
    ...navTree[key]
  });
  Object.entries(navTree).forEach(([key, node]) => {
    var _node$deepLink;
    if (node.getIsActive && location) {
      const isActive = node.getIsActive({
        pathNameSerialized: currentPathname,
        location,
        prepend
      });
      if (isActive) {
        const keysWithParents = extractParentPaths(key, navTree);
        activeNodes.push(keysWithParents.map(activeNodeFromKey));
      }
      return;
    }
    const nodePath = serializeDeeplinkUrl((_node$deepLink = node.deepLink) === null || _node$deepLink === void 0 ? void 0 : _node$deepLink.url);
    if (nodePath) {
      const match = currentPathname.startsWith(nodePath);
      if (match) {
        const {
          length
        } = nodePath;
        if (!matches[length]) {
          matches[length] = [];
        }
        matches[length].push(key);
        // If there are multiple node matches of the same URL path length, we want to order them by
        // tree depth, so that the longest match (deepest node) comes first.
        matches[length].sort((a, b) => b.length - a.length);
      }
    }
  });
  if (matches.length > 0) {
    const longestMatch = matches[matches.length - 1];
    longestMatch.forEach(key => {
      const keysWithParents = extractParentPaths(key, navTree);
      activeNodes.push(keysWithParents.map(activeNodeFromKey));
    });
  }
  return activeNodes;
};
exports.findActiveNodes = findActiveNodes;
let uniqueId = 0;
function generateUniqueNodeId() {
  const id = `node${uniqueId++}`;
  return id;
}
function isAbsoluteLink(link) {
  return link.startsWith('http://') || link.startsWith('https://');
}
function getNavigationNodeId({
  id: _id,
  link
}, idGenerator = generateUniqueNodeId) {
  const id = _id !== null && _id !== void 0 ? _id : link;
  return id !== null && id !== void 0 ? id : idGenerator();
}
function getNavigationNodeHref({
  href,
  deepLink
}) {
  var _deepLink$url;
  return (_deepLink$url = deepLink === null || deepLink === void 0 ? void 0 : deepLink.url) !== null && _deepLink$url !== void 0 ? _deepLink$url : href;
}

/**
 * We don't have currently a way to know if a user has access to a Cloud section.
 * TODO: This function will have to be revisited once we have an API from Cloud to know the user
 * permissions.
 */
function hasUserAccessToCloudLink() {
  return true;
}
function getNodeStatus({
  link,
  deepLink,
  cloudLink,
  sideNavStatus
}, {
  cloudLinks
}) {
  if (link && !deepLink) {
    // If a link is provided, but no deepLink is found, don't render anything
    return 'remove';
  }
  if (cloudLink) {
    if (!cloudLinks[cloudLink]) {
      // Invalid cloudLinkId or link url has not been set in kibana.yml
      return 'remove';
    }
    if (!hasUserAccessToCloudLink()) return 'remove';
  }
  return sideNavStatus !== null && sideNavStatus !== void 0 ? sideNavStatus : 'visible';
}
function getTitleForNode(navNode, {
  deepLink,
  cloudLinks
}) {
  if (navNode.title) {
    return navNode.title;
  }
  if (deepLink !== null && deepLink !== void 0 && deepLink.title) {
    return deepLink.title;
  }
  if (navNode.cloudLink) {
    var _cloudLinks$navNode$c, _cloudLinks$navNode$c2;
    return (_cloudLinks$navNode$c = (_cloudLinks$navNode$c2 = cloudLinks[navNode.cloudLink]) === null || _cloudLinks$navNode$c2 === void 0 ? void 0 : _cloudLinks$navNode$c2.title) !== null && _cloudLinks$navNode$c !== void 0 ? _cloudLinks$navNode$c : '';
  }
  return; // title is optional in EuiCollapsibleNavItemProps
}
function validateNodeProps({
  id,
  link,
  href,
  cloudLink,
  renderAs
}) {
  if (link && cloudLink) {
    throw new Error(`[Chrome navigation] Error in node [${id}]. Only one of "link" or "cloudLink" can be provided.`);
  }
  if (href && cloudLink) {
    throw new Error(`[Chrome navigation] Error in node [${id}]. Only one of "href" or "cloudLink" can be provided.`);
  }
}
const initNavNode = (node, {
  cloudLinks,
  deepLinks,
  parentNodePath,
  index = 0
}) => {
  var _cloudLinks$cloudLink;
  validateNodeProps(node);
  const {
    cloudLink,
    link,
    children,
    ...navNodeFromProps
  } = node;
  const deepLink = link !== undefined ? deepLinks[link] : undefined;
  const sideNavStatus = getNodeStatus({
    link,
    deepLink,
    cloudLink,
    sideNavStatus: navNodeFromProps.sideNavStatus
  }, {
    cloudLinks
  });
  if (sideNavStatus === 'remove') {
    return null;
  }
  const id = getNavigationNodeId(node, () => `node-${index}`);
  const title = getTitleForNode(node, {
    deepLink,
    cloudLinks
  });
  const isExternalLink = cloudLink != null;
  const href = isExternalLink ? (_cloudLinks$cloudLink = cloudLinks[cloudLink]) === null || _cloudLinks$cloudLink === void 0 ? void 0 : _cloudLinks$cloudLink.href : node.href;
  const path = parentNodePath ? `${parentNodePath}.${id}` : id;
  if (href && !isAbsoluteLink(href)) {
    throw new Error(`href must be an absolute URL. Node id [${id}].`);
  }
  const navNode = {
    ...navNodeFromProps,
    id,
    href: getNavigationNodeHref({
      href,
      deepLink
    }),
    path,
    title,
    deepLink,
    isExternalLink,
    sideNavStatus
  };
  return navNode;
};
const parseNavigationTree = (id, navigationTreeDef, {
  deepLinks,
  cloudLinks
}) => {
  var _navigationTreeDef$bo, _navigationTreeDef$bo2;
  // The navigation tree that represents the global navigation and will be used by the Chrome service
  const navigationTree = [];

  // Contains UI layout information (body, footer) and render "special" blocks like recently accessed.
  const navigationTreeUI = {
    id,
    body: []
  };
  const initNodeAndChildren = (node, {
    index = 0,
    parentPath = []
  } = {}) => {
    const navNode = initNavNode(node, {
      cloudLinks,
      deepLinks,
      parentNodePath: parentPath.length > 0 ? parentPath.join('.') : undefined,
      index
    });
    if (navNode && node.children) {
      navNode.children = node.children.map((child, i) => initNodeAndChildren(child, {
        index: i,
        parentPath: [...parentPath, navNode.id]
      })).filter(child => child !== null);
    }
    return navNode;
  };
  const onNodeInitiated = (navNode, section = 'body') => {
    if (navNode) {
      // Add the node to the global navigation tree
      navigationTree.push(navNode);

      // Add the node to the Side Navigation UI tree
      if (!navigationTreeUI[section]) {
        navigationTreeUI[section] = [];
      }
      navigationTreeUI[section].push(navNode);
    }
  };
  const parseNodesArray = (nodes, section = 'body', startIndex = 0) => {
    if (!nodes) return;
    nodes.forEach((node, i) => {
      const navNode = initNodeAndChildren(node, {
        index: startIndex + i
      });
      onNodeInitiated(navNode, section);
    });
  };
  parseNodesArray(navigationTreeDef.body, 'body');
  parseNodesArray(navigationTreeDef.footer, 'footer', (_navigationTreeDef$bo = (_navigationTreeDef$bo2 = navigationTreeDef.body) === null || _navigationTreeDef$bo2 === void 0 ? void 0 : _navigationTreeDef$bo2.length) !== null && _navigationTreeDef$bo !== void 0 ? _navigationTreeDef$bo : 0);
  return {
    navigationTree,
    navigationTreeUI
  };
};
exports.parseNavigationTree = parseNavigationTree;