"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AgentlessConnectorDeploymentsSyncService = void 0;
exports.infraSyncTaskRunner = infraSyncTaskRunner;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _services = require("./services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const AGENTLESS_CONNECTOR_DEPLOYMENTS_SYNC_TASK_ID = 'search:agentless-connectors-manager-task';
const AGENTLESS_CONNECTOR_DEPLOYMENTS_SYNC_TASK_TYPE = 'search:agentless-connectors-manager';
const SCHEDULE = {
  interval: '1m'
};
function infraSyncTaskRunner(logger, getStartServices, agentlessConnectorsInfraServiceFactory) {
  return ({
    taskInstance
  }) => {
    return {
      run: async () => {
        try {
          const service = agentlessConnectorsInfraServiceFactory.getAgentlessConnectorsInfraService();
          if (!service) {
            logger.warn('Agentless connectors infra service not initialized');
            return {
              state: {},
              schedule: SCHEDULE
            };
          }
          // We fetch some info even if license does not permit actual operations.
          // This is done so that we could give a warning to the user only
          // if they are actually using the feature.
          logger.debug('Checking state of connectors and agentless policies');

          // Fetch connectors
          const nativeConnectors = await service.getNativeConnectors();
          const policiesMetadata = await service.getConnectorPackagePolicies();

          // Check license if any native connectors or agentless policies found
          if (nativeConnectors.length > 0 || policiesMetadata.length > 0) {
            const [_core, start] = await getStartServices();
            const license = await start.licensing.getLicense();
            if (license.check('fleet', 'platinum').state !== 'valid') {
              logger.warn('Current license is not compatible with agentless connectors. Please upgrade the license to at least platinum');
              return;
            }
          }

          // Deploy Policies
          const connectorsToDeploy = (0, _services.getConnectorsToDeploy)(policiesMetadata, nativeConnectors);
          let agentlessConnectorsDeployed = 0;
          for (const connectorMetadata of connectorsToDeploy) {
            // We try-catch to still be able to deploy other connectors if some fail
            try {
              await service.deployConnector(connectorMetadata);
              agentlessConnectorsDeployed += 1;
            } catch (e) {
              logger.warn(`Error creating an agentless deployment for connector ${connectorMetadata.id}: ${e.message}`);
            }
          }

          // Delete policies
          const policiesToDelete = (0, _services.getPoliciesToDelete)(policiesMetadata, nativeConnectors);
          let agentlessConnectorsRemoved = 0;
          for (const policyMetadata of policiesToDelete) {
            // We try-catch to still be able to deploy other connectors if some fail
            try {
              await service.removeDeployment(policyMetadata.package_policy_id);
              agentlessConnectorsRemoved += 1;
            } catch (e) {
              logger.warn(`Error when deleting a package policy ${policyMetadata.package_policy_id}: ${e.message}`);
            }
          }
          return {
            state: {},
            schedule: SCHEDULE
          };
        } catch (e) {
          logger.warn(`Error executing agentless deployment sync task: ${e.message}`);
          return {
            state: {},
            schedule: SCHEDULE
          };
        }
      },
      cancel: async () => {
        logger.warn('timed out');
      }
    };
  };
}
class AgentlessConnectorDeploymentsSyncService {
  constructor(logger) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    this.logger = logger;
  }
  registerInfraSyncTask(core, plugins, agentlessConnectorsInfraServiceFactory) {
    plugins.taskManager.registerTaskDefinitions({
      [AGENTLESS_CONNECTOR_DEPLOYMENTS_SYNC_TASK_TYPE]: {
        title: 'Agentless Connector Deployment Manager',
        description: 'This task peridocally checks native connectors, agent policies and syncs them if they are out of sync',
        timeout: '1m',
        maxAttempts: 3,
        createTaskRunner: infraSyncTaskRunner(this.logger, core.getStartServices, agentlessConnectorsInfraServiceFactory)
      }
    });
  }
  async scheduleInfraSyncTask(config, taskManager) {
    this.logger.info(`Scheduling ${AGENTLESS_CONNECTOR_DEPLOYMENTS_SYNC_TASK_ID}`);
    try {
      var _taskInstance$schedul;
      const taskInstance = await taskManager.ensureScheduled({
        id: AGENTLESS_CONNECTOR_DEPLOYMENTS_SYNC_TASK_ID,
        taskType: AGENTLESS_CONNECTOR_DEPLOYMENTS_SYNC_TASK_TYPE,
        schedule: SCHEDULE,
        params: {},
        state: {},
        scope: ['search', 'connectors']
      });
      this.logger.info(`Scheduled ${AGENTLESS_CONNECTOR_DEPLOYMENTS_SYNC_TASK_ID} with interval ${(_taskInstance$schedul = taskInstance.schedule) === null || _taskInstance$schedul === void 0 ? void 0 : _taskInstance$schedul.interval}`);
      return taskInstance;
    } catch (e) {
      this.logger.error(`Error scheduling ${AGENTLESS_CONNECTOR_DEPLOYMENTS_SYNC_TASK_ID}, received ${e.message}`);
      return null;
    }
  }
}
exports.AgentlessConnectorDeploymentsSyncService = AgentlessConnectorDeploymentsSyncService;