"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AutomaticImportSavedObjectService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/core/server");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class AutomaticImportSavedObjectService {
  constructor(logger, savedObjectsClient, security) {
    (0, _defineProperty2.default)(this, "savedObjectsClient", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "security", null);
    this.security = security;
    this.logger = logger.get('savedObjectsService');
    this.savedObjectsClient = savedObjectsClient;
  }

  /**
   * Helper function to parse and increment a semantic version string (x.y.z)
   * @param currentVersion - Current semantic version string (e.g., "1.0.0")
   * @param incrementType - Optional: Which part to increment: 'major' | 'minor' | 'patch'. Defaults to 'patch'.
   * @returns Incremented semantic version string
   */
  incrementSemanticVersion(currentVersion, incrementType = 'patch') {
    if (!currentVersion) {
      return '1.0.0';
    }
    const versionParts = currentVersion.split('.');
    if (versionParts.length !== 3) {
      throw new Error('Invalid version format');
    }
    let [major, minor, patch] = versionParts.map(v => parseInt(v, 10));
    if (isNaN(major) || isNaN(minor) || isNaN(patch)) {
      return '1.0.0';
    }
    switch (incrementType) {
      case 'major':
        major += 1;
        minor = 0;
        patch = 0;
        break;
      case 'minor':
        minor += 1;
        patch = 0;
        break;
      case 'patch':
      default:
        patch += 1;
        break;
    }
    return `${major}.${minor}.${patch}`;
  }

  /**
   * Integration Operations
   */

  /**
   * Create an integration that may or may not already have data streams associated with it.
   * @param request - The Kibana request object
   * @param data - The integration data. Must include an integration_id.
   * @param options - The options for the create
   * @returns The saved object
   */
  async insertIntegration(request, data, options) {
    var _this$security;
    const authenticatedUser = (_this$security = this.security) === null || _this$security === void 0 ? void 0 : _this$security.authc.getCurrentUser(request);
    if (!authenticatedUser) {
      throw new Error('No user authenticated');
    }
    const {
      integration_id: integrationId,
      data_stream_count: dataStreamCount = 0,
      metadata = {}
    } = data;
    if (!integrationId) {
      throw new Error('Integration ID is required');
    }
    try {
      this.logger.debug(`Creating integration: ${integrationId}`);
      const initialIntegrationData = {
        integration_id: integrationId,
        data_stream_count: dataStreamCount,
        created_by: authenticatedUser.username,
        status: _constants.TASK_STATUSES.pending,
        metadata: {
          ...metadata,
          created_at: new Date().toISOString(),
          version: '1.0.0'
        }
      };
      return await this.savedObjectsClient.create(_constants.INTEGRATION_SAVED_OBJECT_TYPE, initialIntegrationData, {
        // overwrite id that is default generated by the saved objects service
        id: integrationId,
        ...options
      });
    } catch (error) {
      // Create will throw a confict if integration ID already exists.
      if (_server.SavedObjectsErrorHelpers.isConflictError(error)) {
        throw new Error(`Integration ${integrationId} already exists`);
      }
      this.logger.error(`Failed to create integration: ${error}`);
      throw error;
    }
  }

  /**
   * Create or update an integration
   * @param data - The integration data. Must include an integration_id.
   * @param expectedVersion - The expected version for optimistic concurrency control at the application layer. Required to ensure data consistency.
   * @param versionUpdate - Optional: specify which version part to increment ('major' | 'minor' | 'patch'). Defaults to incrementing 'patch'.
   * @param options - The options for the update.
   * @returns The saved object
   */
  async updateIntegration(data, expectedVersion, versionUpdate, options) {
    const {
      integration_id: integrationId,
      data_stream_count: dataStreamCount = 0,
      status,
      metadata = {}
    } = data;
    if (!integrationId) {
      throw new Error('Integration ID is required');
    }
    try {
      var _existingIntegration$;
      this.logger.debug(`Updating integration: ${integrationId}`);
      const existingIntegration = await this.getIntegration(integrationId);
      if (!existingIntegration) {
        throw new Error(`Integration ${integrationId} not found`);
      }
      const currentVersion = ((_existingIntegration$ = existingIntegration.attributes.metadata) === null || _existingIntegration$ === void 0 ? void 0 : _existingIntegration$.version) || '1.0.0';
      if (currentVersion !== expectedVersion) {
        throw new Error(`Version conflict: Integration ${integrationId} has been updated. Expected version ${expectedVersion}, but current version is ${currentVersion}. Please fetch the latest version and try again.`);
      }
      const newVersion = this.incrementSemanticVersion(currentVersion, versionUpdate);
      const integrationData = {
        integration_id: integrationId,
        data_stream_count: dataStreamCount,
        created_by: existingIntegration.attributes.created_by,
        status: status || existingIntegration.attributes.status,
        metadata: {
          ...metadata,
          version: newVersion
        }
      };
      const internalVersion = existingIntegration.version;
      return await this.savedObjectsClient.update(_constants.INTEGRATION_SAVED_OBJECT_TYPE, integrationId, integrationData, {
        ...options,
        version: internalVersion
      });
    } catch (error) {
      this.logger.error(`Failed to update integration: ${error}`);
      throw error;
    }
  }

  /**
   * Get an integration by ID
   * @param integrationId - The ID of the integration
   * @returns The integration
   */
  async getIntegration(integrationId) {
    try {
      this.logger.debug(`Getting integration: ${integrationId}`);
      return await this.savedObjectsClient.get(_constants.INTEGRATION_SAVED_OBJECT_TYPE, integrationId);
    } catch (error) {
      this.logger.error(`Failed to get integration ${integrationId}: ${error}`);
      throw error;
    }
  }

  /**
   * @returns All integrations
   */
  async getAllIntegrations() {
    try {
      this.logger.debug('Getting all integrations');
      return await this.savedObjectsClient.find({
        type: _constants.INTEGRATION_SAVED_OBJECT_TYPE
      });
    } catch (error) {
      this.logger.error(`Failed to get all integrations: ${error}`);
      throw error;
    }
  }

  /**
   * Delete an integration by ID and cascade delete all associated data streams
   * @param integrationId - The ID of the integration
   * @param options - The options for the delete
   * @returns Object containing deletion results with success status and any errors
   */
  async deleteIntegration(integrationId, options) {
    this.logger.debug(`Starting cascade deletion for integration: ${integrationId}`);
    const deletionErrors = [];
    let dataStreamsDeleted = 0;
    try {
      // delete up to 100 data streams at a time
      const perPage = 100;
      let page = 1;
      let hasMore = true;
      const dataStreamsToDelete = [];
      while (hasMore) {
        try {
          const dataStreamsResponse = await this.savedObjectsClient.find({
            type: _constants.DATA_STREAM_SAVED_OBJECT_TYPE,
            filter: `${_constants.DATA_STREAM_SAVED_OBJECT_TYPE}.attributes.integration_id: ${JSON.stringify(integrationId)}`,
            page,
            perPage,
            fields: ['integration_id']
          });
          dataStreamsToDelete.push(...dataStreamsResponse.saved_objects.map(ds => ({
            id: ds.id,
            type: _constants.DATA_STREAM_SAVED_OBJECT_TYPE
          })));

          // If 100 saved objects are retrieved in a page, then we try to see if there are more
          hasMore = dataStreamsResponse.saved_objects.length === perPage;
          page++;
        } catch (error) {
          this.logger.error(`Failed to find data streams for integration ${integrationId} on page ${page}: ${error}`);
          hasMore = false;
        }
      }
      this.logger.debug(`Found ${dataStreamsToDelete.length} data streams to delete for integration ${integrationId}`);
      if (dataStreamsToDelete.length > 0) {
        for (let i = 0; i < dataStreamsToDelete.length; i += _constants.BULK_DELETE_CHUNK_SIZE) {
          const chunk = dataStreamsToDelete.slice(i, i + _constants.BULK_DELETE_CHUNK_SIZE);
          try {
            const bulkDeleteResult = await this.savedObjectsClient.bulkDelete(chunk, {
              ...options,
              force: true
            });

            // Process results to track successes and failures
            bulkDeleteResult.statuses.forEach(status => {
              if (status.success) {
                dataStreamsDeleted++;
              } else {
                var _status$error;
                const errorMessage = ((_status$error = status.error) === null || _status$error === void 0 ? void 0 : _status$error.message) || 'Unknown error';
                this.logger.warn(`Failed to delete data stream ${status.id}: ${errorMessage}`);
                deletionErrors.push({
                  id: status.id,
                  error: errorMessage
                });
              }
            });
          } catch (bulkDeleteError) {
            // If bulk delete fails entirely, log and track errors for this chunk
            this.logger.error(`Bulk delete failed for chunk starting at index ${i}: ${bulkDeleteError}`);
            chunk.forEach(ds => {
              deletionErrors.push({
                id: ds.id,
                error: bulkDeleteError instanceof Error ? bulkDeleteError.message : String(bulkDeleteError)
              });
            });
          }
        }
        this.logger.debug(`Deleted ${dataStreamsDeleted} of ${dataStreamsToDelete.length} data streams for integration ${integrationId}`);
      }

      // Delete the integration if we successfully deleted all data streams OR the a force option is provided
      const allDataStreamsDeleted = deletionErrors.length === 0;
      if (!allDataStreamsDeleted && !(options !== null && options !== void 0 && options.force)) {
        throw new Error(`Cannot delete integration ${integrationId}: Failed to delete ${deletionErrors.length} data streams. ` + `Use force option to delete the integration anyway. Errors: ${JSON.stringify(deletionErrors)}`);
      }
      await this.savedObjectsClient.delete(_constants.INTEGRATION_SAVED_OBJECT_TYPE, integrationId, options);
      this.logger.info(`Successfully deleted integration ${integrationId} and ${dataStreamsDeleted} associated data streams`);
      return {
        success: true,
        dataStreamsDeleted,
        errors: deletionErrors
      };
    } catch (error) {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(error)) {
        this.logger.error(`Integration ${integrationId} not found`);
        throw error;
      }
      this.logger.error(`Failed to delete integration ${integrationId}: ${error}`);
      throw error;
    }
  }

  /**
   * Data Stream Operations
   */

  /**
   * Create a data stream
   * @param request - The Kibana request object
   * @param data - The data stream data. Must include an integration_id and data_stream_id.
   * @param options - The options for the create
   * @returns The created data stream
   */
  async insertDataStream(request, data, options) {
    var _this$security2;
    const authenticatedUser = (_this$security2 = this.security) === null || _this$security2 === void 0 ? void 0 : _this$security2.authc.getCurrentUser(request);
    if (!authenticatedUser) {
      throw new Error('No user authenticated');
    }
    const {
      integration_id: integrationId,
      data_stream_id: dataStreamId,
      job_info: jobInfo = {
        status: _constants.TASK_STATUSES.pending,
        job_id: '',
        job_type: ''
      },
      metadata,
      result = {}
    } = data;
    if (!integrationId) {
      throw new Error('Integration ID is required');
    }
    if (!dataStreamId) {
      throw new Error('Data stream ID is required');
    }
    let existingIntegration = null;
    // Check for existing integration
    try {
      existingIntegration = await this.getIntegration(integrationId);
      this.logger.debug(`Integration ${integrationId} found, will update count after data stream creation`);
    } catch (error) {
      this.logger.debug(`Integration ${integrationId} not found, will create after data stream creation`);
    }

    // Create the data stream first since we need it before we need to create a missing integration
    try {
      this.logger.debug(`Creating data stream: ${dataStreamId}`);
      const {
        sample_count: sampleCount = 0,
        ...restMetadata
      } = metadata || {};
      const initialDataStreamData = {
        integration_id: integrationId,
        data_stream_id: dataStreamId,
        created_by: authenticatedUser.username,
        job_info: jobInfo,
        metadata: {
          sample_count: sampleCount,
          ...restMetadata,
          created_at: new Date().toISOString(),
          version: '1.0.0'
        },
        result: result || {}
      };

      // Create automatically checks if there is an existing data stream with the same id and will throw an error if so
      const createdDataStream = await this.savedObjectsClient.create(_constants.DATA_STREAM_SAVED_OBJECT_TYPE, initialDataStreamData, {
        ...options,
        id: dataStreamId
      });

      // After successful data stream creation, update data stream count in the integration
      // or create a new integration if it doesn't exist
      try {
        if (existingIntegration) {
          var _existingIntegration$2;
          this.logger.debug(`Data stream created successfully, incrementing integration ${integrationId} count`);
          const updatedIntegrationData = {
            ...existingIntegration.attributes,
            data_stream_count: existingIntegration.attributes.data_stream_count + 1
          };
          await this.updateIntegration(updatedIntegrationData, ((_existingIntegration$2 = existingIntegration.attributes.metadata) === null || _existingIntegration$2 === void 0 ? void 0 : _existingIntegration$2.version) || '1.0.0');
        } else {
          this.logger.debug(`Data stream created successfully, creating new integration ${integrationId}`);
          const defaultIntegrationData = {
            integration_id: integrationId,
            data_stream_count: 1,
            created_by: authenticatedUser.username,
            status: _constants.TASK_STATUSES.pending,
            metadata: {
              created_at: new Date().toISOString(),
              version: '1.0.0',
              title: `Auto-generated integration ${integrationId}`
            }
          };
          await this.insertIntegration(request, defaultIntegrationData);
        }
      } catch (integrationError) {
        this.logger.error(`Failed to create integration ${integrationId} after creating data stream ${dataStreamId}: ${integrationError}`);
      }
      return createdDataStream;
    } catch (error) {
      if (_server.SavedObjectsErrorHelpers.isConflictError(error)) {
        throw new Error(`Data stream ${dataStreamId} already exists`);
      }
      throw error;
    }
  }

  /**
   * Update a data stream
   * @param data - The data stream data. Must include an integration_id and data_stream_id.
   * @param expectedVersion - The expected version for optimistic concurrency control at the application layer. Required to ensure data consistency.
   * @param versionUpdate - Optional: specify which version part to increment ('major' | 'minor' | 'patch').  Defaults to incrementing 'patch'.
   * @param options - The options for the update.
   * @returns The updated data stream
   */
  async updateDataStream(data, expectedVersion, versionUpdate, options) {
    const {
      integration_id: integrationId,
      data_stream_id: dataStreamId,
      job_info: jobInfo,
      metadata = {
        sample_count: 0
      },
      result = {}
    } = data;
    if (!integrationId) {
      throw new Error('Integration ID is required');
    }
    if (!dataStreamId) {
      throw new Error('Data stream ID is required');
    }
    try {
      var _existingDataStream$a;
      this.logger.debug(`Updating data stream: ${dataStreamId}`);

      // A Data Stream must always be associated with an Integration
      const integrationTarget = await this.getIntegration(integrationId);
      if (!integrationTarget) {
        throw new Error(`Integration associated with this data stream ${integrationId} not found`);
      }
      const existingDataStream = await this.getDataStream(dataStreamId);
      const currentVersion = ((_existingDataStream$a = existingDataStream.attributes.metadata) === null || _existingDataStream$a === void 0 ? void 0 : _existingDataStream$a.version) || '1.0.0';
      if (currentVersion !== expectedVersion) {
        throw new Error(`Version conflict: Data stream ${dataStreamId} has been updated. Expected version ${expectedVersion}, but current version is ${currentVersion}. Please fetch the latest version and try again.`);
      }
      const newVersion = this.incrementSemanticVersion(currentVersion, versionUpdate);
      const dataStreamData = {
        integration_id: integrationId,
        data_stream_id: dataStreamId,
        created_by: existingDataStream.attributes.created_by,
        job_info: jobInfo,
        metadata: {
          ...metadata,
          created_at: existingDataStream.attributes.metadata.created_at,
          version: newVersion
        },
        result: result || {}
      };
      const internalVersion = existingDataStream.version;
      return await this.savedObjectsClient.update(_constants.DATA_STREAM_SAVED_OBJECT_TYPE, dataStreamId, dataStreamData, {
        ...options,
        version: internalVersion
      });
    } catch (error) {
      if (_server.SavedObjectsErrorHelpers.isConflictError(error)) {
        throw new Error(`Data stream ${dataStreamId} has been updated since you last fetched it. Please fetch the latest version and try again.`);
      }
      this.logger.error(`Failed to update data stream: ${error}`);
      throw error;
    }
  }

  /**
   * Get a data stream by ID
   * @param dataStreamId - The ID of the data stream
   * @returns The data stream
   */
  async getDataStream(dataStreamId) {
    try {
      this.logger.debug(`Getting data stream: ${dataStreamId}`);
      return await this.savedObjectsClient.get(_constants.DATA_STREAM_SAVED_OBJECT_TYPE, dataStreamId);
    } catch (error) {
      this.logger.error(`Failed to get data stream ${dataStreamId}: ${error}`);
      throw error;
    }
  }

  /**
   * Get all data streams
   * @returns All data streams
   */
  async getAllDataStreams() {
    try {
      this.logger.debug('Getting all data streams');
      return await this.savedObjectsClient.find({
        type: _constants.DATA_STREAM_SAVED_OBJECT_TYPE
      });
    } catch (error) {
      this.logger.error(`Failed to get all data streams: ${error}`);
      throw error;
    }
  }

  /**
   * Find all data streams by integration ID
   * @param integrationId - The ID of the integration
   * @returns All data streams for the integration
   */
  async findAllDataStreamsByIntegrationId(integrationId) {
    try {
      this.logger.debug(`Finding all data streams for integration: ${integrationId}`);
      return await this.savedObjectsClient.find({
        type: _constants.DATA_STREAM_SAVED_OBJECT_TYPE,
        filter: `${_constants.DATA_STREAM_SAVED_OBJECT_TYPE}.attributes.integration_id: ${JSON.stringify(integrationId)}`
      });
    } catch (error) {
      this.logger.error(`Failed to find all data streams for integration ${integrationId}: ${error}`);
      throw error;
    }
  }

  /**
   * Delete a data stream by ID
   * @param dataStreamId - The ID of the data stream
   * @param options - The options for the delete
   * @returns The deleted data stream
   */
  async deleteDataStream(dataStreamId, options) {
    try {
      this.logger.debug(`Deleting data stream with id:${dataStreamId}`);

      // Get the data stream to find its parent integration
      const dataStream = await this.getDataStream(dataStreamId);
      const parentIntegrationId = dataStream.attributes.integration_id;

      // Delete the data stream
      await this.savedObjectsClient.delete(_constants.DATA_STREAM_SAVED_OBJECT_TYPE, dataStreamId, options);

      // Decrement the data stream count in the parent integration
      try {
        var _parentIntegration$at;
        const parentIntegration = await this.getIntegration(parentIntegrationId);
        if (!parentIntegration) {
          throw new Error(`Integration associated with this data stream ${parentIntegrationId} not found`);
        }
        const updatedIntegrationData = {
          ...parentIntegration.attributes,
          data_stream_count: parentIntegration.attributes.data_stream_count - 1
        };
        await this.updateIntegration(updatedIntegrationData, ((_parentIntegration$at = parentIntegration.attributes.metadata) === null || _parentIntegration$at === void 0 ? void 0 : _parentIntegration$at.version) || '1.0.0');
      } catch (integrationError) {
        this.logger.error(`Failed to update integration ${parentIntegrationId} after deleting data stream ${dataStreamId}: ${integrationError}`);
      }
    } catch (error) {
      this.logger.error(`Failed to delete data stream ${dataStreamId}: ${error}`);
      throw error;
    }
  }
}
exports.AutomaticImportSavedObjectService = AutomaticImportSavedObjectService;