"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getUnifiedTraceErrors = getUnifiedTraceErrors;
exports.requiredOtelFields = exports.optionalOtelFields = void 0;
var _utils = require("@kbn/apm-data-access-plugin/server/utils");
var _server = require("@kbn/observability-plugin/server");
var _apm = require("../../../common/es_fields/apm");
var _as_mutable_array = require("../../../common/utils/as_mutable_array");
var _get_apm_trace_error_query = require("./get_apm_trace_error_query");
var _compact_map = require("../../utils/compact_map");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getUnifiedTraceErrors({
  apmEventClient,
  logsClient,
  traceId,
  docId,
  start,
  end
}) {
  const commonParams = {
    traceId,
    docId,
    start,
    end
  };
  const [apmErrors, unprocessedOtelErrors] = await Promise.all([getUnifiedApmTraceError({
    apmEventClient,
    ...commonParams
  }), getUnprocessedOtelErrors({
    logsClient,
    ...commonParams
  })]);
  return {
    apmErrors,
    unprocessedOtelErrors,
    totalErrors: apmErrors.length + unprocessedOtelErrors.length
  };
}
const requiredOtelFields = exports.requiredOtelFields = (0, _as_mutable_array.asMutableArray)([_apm.SPAN_ID, _apm.ID]);
const optionalOtelFields = exports.optionalOtelFields = (0, _as_mutable_array.asMutableArray)([_apm.EXCEPTION_TYPE, _apm.EXCEPTION_MESSAGE, _apm.TIMESTAMP_US, _apm.OTEL_EVENT_NAME]);
async function getUnifiedApmTraceError(params) {
  const response = await (0, _get_apm_trace_error_query.getApmTraceErrorQuery)(params);
  return (0, _compact_map.compactMap)(response.hits.hits, hit => {
    var _event$TRANSACTION_ID, _errorSource$error$ex, _errorSource$error$ex2;
    const errorSource = 'error' in hit._source ? hit._source : undefined;
    const event = hit.fields ? (0, _utils.accessKnownApmEventFields)(hit.fields).requireFields(_get_apm_trace_error_query.requiredFields) : undefined;
    const spanId = (_event$TRANSACTION_ID = event === null || event === void 0 ? void 0 : event[_apm.TRANSACTION_ID]) !== null && _event$TRANSACTION_ID !== void 0 ? _event$TRANSACTION_ID : event === null || event === void 0 ? void 0 : event[_apm.SPAN_ID];
    if (!event || !spanId) {
      return null;
    }
    const error = {
      id: event[_apm.ID],
      spanId,
      timestamp: {
        us: event[_apm.TIMESTAMP_US]
      },
      error: {
        id: event[_apm.ERROR_ID],
        grouping_key: event[_apm.ERROR_GROUP_ID],
        log: errorSource === null || errorSource === void 0 ? void 0 : errorSource.error.log,
        culprit: event[_apm.ERROR_CULPRIT],
        exception: (_errorSource$error$ex = errorSource === null || errorSource === void 0 ? void 0 : (_errorSource$error$ex2 = errorSource.error.exception) === null || _errorSource$error$ex2 === void 0 ? void 0 : _errorSource$error$ex2[0]) !== null && _errorSource$error$ex !== void 0 ? _errorSource$error$ex : {
          type: event[_apm.ERROR_EXC_TYPE],
          message: event[_apm.ERROR_EXC_MESSAGE],
          handled: event[_apm.ERROR_EXC_HANDLED]
        }
      }
    };
    return error;
  });
}
async function getUnprocessedOtelErrors({
  logsClient,
  traceId,
  docId,
  start,
  end
}) {
  const response = await logsClient.search({
    query: {
      bool: {
        filter: [...(0, _server.rangeQuery)(start, end), ...(0, _server.termQuery)(_apm.TRACE_ID, traceId), ...(0, _server.termQuery)(_apm.SPAN_ID, docId)],
        should: [...(0, _server.termQuery)(_apm.OTEL_EVENT_NAME, 'exception'), ...(0, _server.existsQuery)(_apm.EXCEPTION_TYPE), ...(0, _server.existsQuery)(_apm.EXCEPTION_MESSAGE)],
        minimum_should_match: 1,
        must_not: {
          exists: {
            field: _apm.PROCESSOR_EVENT
          }
        }
      }
    },
    fields: [...requiredOtelFields, ...optionalOtelFields]
  });
  return (0, _compact_map.compactMap)(response.hits.hits, hit => {
    const event = hit.fields ? (0, _utils.accessKnownApmEventFields)(hit.fields).requireFields(requiredOtelFields) : undefined;
    if (!event) return null;
    const timestamp = event[_apm.TIMESTAMP_US];
    const error = {
      id: event[_apm.ID],
      spanId: event[_apm.SPAN_ID],
      timestamp: timestamp != null ? {
        us: timestamp
      } : undefined,
      eventName: event[_apm.OTEL_EVENT_NAME],
      error: {
        exception: {
          type: event[_apm.EXCEPTION_TYPE],
          message: event[_apm.EXCEPTION_MESSAGE]
        }
      }
    };
    return error;
  });
}