"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSpanLinksDetails = getSpanLinksDetails;
var _server = require("@kbn/observability-plugin/server");
var _common = require("@kbn/observability-plugin/common");
var _lodash = require("lodash");
var _utils = require("@kbn/apm-data-access-plugin/server/utils");
var _as_mutable_array = require("../../../common/utils/as_mutable_array");
var _apm = require("../../../common/es_fields/apm");
var _utils2 = require("./utils");
var _parse_otel_duration = require("../../lib/helpers/parse_otel_duration");
var _compact_map = require("../../utils/compact_map");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function fetchSpanLinksDetails({
  apmEventClient,
  kuery,
  spanLinks,
  start,
  end
}) {
  const {
    startWithBuffer,
    endWithBuffer
  } = (0, _utils2.getBufferedTimerange)({
    start,
    end
  });
  const requiredFields = (0, _as_mutable_array.asMutableArray)([_apm.TRACE_ID, _apm.SERVICE_NAME]);
  const requiredTxFields = (0, _as_mutable_array.asMutableArray)([_apm.TRANSACTION_ID, _apm.TRANSACTION_NAME, _apm.TRANSACTION_DURATION, _apm.AGENT_NAME]);
  const requiredSpanFields = (0, _as_mutable_array.asMutableArray)([_apm.SPAN_ID, _apm.SPAN_NAME, _apm.SPAN_DURATION, _apm.SPAN_SUBTYPE, _apm.SPAN_TYPE, _apm.AGENT_NAME]);
  const requiredUnprocessedOtelSpanFields = (0, _as_mutable_array.asMutableArray)([_apm.SPAN_ID, _apm.SPAN_NAME, _apm.DURATION, _apm.KIND, _apm.SERVICE_LANGUAGE_NAME]);
  const optionalFields = (0, _as_mutable_array.asMutableArray)([_apm.SERVICE_ENVIRONMENT, _apm.PROCESSOR_EVENT]);
  const response = await apmEventClient.search('get_span_links_details', {
    apm: {
      events: [_common.ProcessorEvent.span, _common.ProcessorEvent.transaction]
    },
    fields: [...requiredFields, ...requiredTxFields, ...requiredSpanFields, ...requiredUnprocessedOtelSpanFields, ...optionalFields],
    track_total_hits: false,
    size: 1000,
    query: {
      bool: {
        filter: [...(0, _server.rangeQuery)(startWithBuffer, endWithBuffer), ...(0, _server.kqlQuery)(kuery), {
          bool: {
            should: spanLinks.map(item => {
              return {
                bool: {
                  filter: [{
                    term: {
                      [_apm.TRACE_ID]: item.trace.id
                    }
                  }, {
                    bool: {
                      should: [{
                        term: {
                          [_apm.SPAN_ID]: item.span.id
                        }
                      }, {
                        term: {
                          [_apm.TRANSACTION_ID]: item.span.id
                        }
                      }],
                      minimum_should_match: 1
                    }
                  }]
                }
              };
            }),
            minimum_should_match: 1
          }
        }]
      }
    }
  }, {
    skipProcessorEventFilter: true
  });
  const spanIdsMap = spanLinks.reduce((acc, link) => {
    acc[link.span.id] = link;
    return acc;
  }, {});
  return (0, _compact_map.compactMap)(response.hits.hits, hit => {
    const commonEvent = (0, _utils.accessKnownApmEventFields)(hit.fields).requireFields(requiredFields);
    const commonDetails = {
      serviceName: commonEvent[_apm.SERVICE_NAME],
      environment: commonEvent[_apm.SERVICE_ENVIRONMENT]
    };
    switch (commonEvent[_apm.PROCESSOR_EVENT]) {
      case _common.ProcessorEvent.transaction:
        {
          const event = commonEvent.requireFields(requiredTxFields);
          return {
            traceId: event[_apm.TRACE_ID],
            spanId: event[_apm.TRANSACTION_ID],
            details: {
              ...commonDetails,
              transactionId: event[_apm.TRANSACTION_ID],
              agentName: event[_apm.AGENT_NAME],
              spanName: event[_apm.TRANSACTION_NAME],
              duration: event[_apm.TRANSACTION_DURATION]
            }
          };
        }
      case _common.ProcessorEvent.span:
        {
          const event = commonEvent.requireFields(requiredSpanFields);

          // The above query might return other spans from the same transaction because siblings spans share the same transaction.id
          // so, if it is a span we need to guarantee that the span.id is the same as the span links ids
          if (!spanIdsMap[event[_apm.SPAN_ID]]) {
            return null;
          }
          return {
            traceId: event[_apm.TRACE_ID],
            spanId: event[_apm.SPAN_ID],
            details: {
              ...commonDetails,
              transactionId: event[_apm.TRANSACTION_ID],
              agentName: event[_apm.AGENT_NAME],
              spanName: event[_apm.SPAN_NAME],
              duration: event[_apm.SPAN_DURATION],
              spanSubtype: event[_apm.SPAN_SUBTYPE],
              spanType: event[_apm.SPAN_TYPE]
            }
          };
        }
      default:
        {
          const event = commonEvent.requireFields(requiredUnprocessedOtelSpanFields);
          return {
            traceId: event[_apm.TRACE_ID],
            spanId: event[_apm.SPAN_ID],
            details: {
              ...commonDetails,
              agentName: event[_apm.SERVICE_LANGUAGE_NAME],
              spanName: event[_apm.SPAN_NAME],
              duration: (0, _parse_otel_duration.parseOtelDuration)(event.duration)
            }
          };
        }
    }
  });
}
async function getSpanLinksDetails({
  apmEventClient,
  spanLinks,
  kuery,
  start,
  end
}) {
  if (!spanLinks.length) {
    return [];
  }

  // chunk span links to avoid too_many_nested_clauses problem
  const spanLinksChunks = (0, _lodash.chunk)(spanLinks, 500);
  const chunkedResponses = await Promise.all(spanLinksChunks.map(spanLinksChunk => fetchSpanLinksDetails({
    apmEventClient,
    kuery,
    spanLinks: spanLinksChunk,
    start,
    end
  })));
  const linkedSpans = chunkedResponses.flat();

  // Creates a map for all span links details found
  const spanLinksDetailsMap = linkedSpans.reduce((acc, spanLink) => {
    const key = `${spanLink.traceId}:${spanLink.spanId}`;
    acc[key] = spanLink;
    return acc;
  }, {});

  // It's important to keep the original order of the span links,
  // so loops trough the original list merging external links and links with details.
  // external links are links that the details were not found in the ES query.
  return (0, _compact_map.compactMap)(spanLinks, item => {
    const key = `${item.trace.id}:${item.span.id}`;
    const details = spanLinksDetailsMap[key];
    if (details) {
      return details;
    }

    // When kuery is not set, we return external links, else we hide this item.
    return !kuery.length ? {
      traceId: item.trace.id,
      spanId: item.span.id
    } : undefined;
  });
}