"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createGetServicesTool = createGetServicesTool;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _tool_schemas = require("./utils/tool_schemas");
var _build_apm_tool_resources = require("./utils/build_apm_tool_resources");
var _get_apm_tool_availability = require("./utils/get_apm_tool_availability");
var _get_apm_service_list = require("../routes/assistant_functions/get_apm_service_list");
var _service_health_status = require("../../common/service_health_status");
var _agent_tool_ids = require("../../common/observability_agent/agent_tool_ids");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getServicesSchema = _zod.z.object({
  ..._tool_schemas.timeRangeSchema.shape,
  serviceEnvironment: _zod.z.string().min(1).optional().describe('Optionally filter the services by the environments that they are running in.'),
  healthStatus: _zod.z.array(_zod.z.enum([_service_health_status.ServiceHealthStatus.unknown, _service_health_status.ServiceHealthStatus.healthy, _service_health_status.ServiceHealthStatus.warning, _service_health_status.ServiceHealthStatus.critical])).optional().describe('Optionally filter the services by their health status.')
});
function createGetServicesTool({
  core,
  plugins,
  logger
}) {
  const toolDefinition = {
    id: _agent_tool_ids.OBSERVABILITY_GET_SERVICES_TOOL_ID,
    type: _onechatCommon.ToolType.builtin,
    description: 'Get the list of monitored services, their health status, and alerts.',
    schema: getServicesSchema,
    tags: ['observability', 'services'],
    availability: {
      cacheMode: 'space',
      handler: async ({
        request
      }) => {
        return (0, _get_apm_tool_availability.getApmToolAvailability)({
          core,
          plugins,
          request,
          logger
        });
      }
    },
    handler: async (args, context) => {
      const {
        request,
        esClient,
        logger: scopedLogger
      } = context;
      try {
        const {
          apmEventClient,
          randomSampler,
          mlClient,
          apmAlertsClient
        } = await (0, _build_apm_tool_resources.buildApmToolResources)({
          core,
          plugins,
          request,
          esClient,
          logger: scopedLogger
        });
        const services = await (0, _get_apm_service_list.getApmServiceList)({
          apmEventClient,
          apmAlertsClient,
          randomSampler,
          mlClient,
          logger: scopedLogger,
          arguments: args
        });
        return {
          results: [{
            type: _tool_result.ToolResultType.other,
            data: {
              services
            }
          }]
        };
      } catch (error) {
        logger.error(`Error getting services: ${error.message}`);
        logger.debug(error);
        return {
          results: [{
            type: _tool_result.ToolResultType.error,
            data: {
              message: `Failed to fetch services: ${error.message}`,
              stack: error.stack
            }
          }]
        };
      }
    }
  };
  return toolDefinition;
}