"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TELEMETRY_TASK_TYPE = exports.TASK_ID = exports.SCHEDULE = void 0;
exports.initializeAlertingTelemetry = initializeAlertingTelemetry;
exports.scheduleAlertingTelemetry = scheduleAlertingTelemetry;
exports.telemetryTaskRunner = telemetryTaskRunner;
var _common = require("@kbn/maintenance-windows-plugin/common");
var _get_telemetry_from_task_manager = require("./lib/get_telemetry_from_task_manager");
var _get_telemetry_from_kibana = require("./lib/get_telemetry_from_kibana");
var _get_telemetry_from_alerts = require("./lib/get_telemetry_from_alerts");
var _get_telemetry_from_event_log = require("./lib/get_telemetry_from_event_log");
var _get_backfill_telemetry = require("./lib/get_backfill_telemetry");
var _task_state = require("./task_state");
var _saved_objects = require("../saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TELEMETRY_TASK_TYPE = exports.TELEMETRY_TASK_TYPE = 'alerting_telemetry';
const TASK_ID = exports.TASK_ID = `Alerting-${TELEMETRY_TASK_TYPE}`;
const SCHEDULE = exports.SCHEDULE = {
  interval: '1d'
};
function initializeAlertingTelemetry(logger, core, taskManager, eventLogIndex) {
  registerAlertingTelemetryTask(logger, core, taskManager, eventLogIndex);
}
function registerAlertingTelemetryTask(logger, core, taskManager, eventLogIndex) {
  taskManager.registerTaskDefinitions({
    [TELEMETRY_TASK_TYPE]: {
      title: 'Alerting usage fetch task',
      timeout: '5m',
      stateSchemaByVersion: _task_state.stateSchemaByVersion,
      createTaskRunner: telemetryTaskRunner(logger, core, eventLogIndex, taskManager.index)
    }
  });
}
function scheduleAlertingTelemetry(logger, taskManager) {
  if (taskManager) {
    scheduleTasks(logger, taskManager).catch(() => {}); // it shouldn't reject, but just in case
  }
}
async function scheduleTasks(logger, taskManager) {
  try {
    await taskManager.ensureScheduled({
      id: TASK_ID,
      taskType: TELEMETRY_TASK_TYPE,
      state: _task_state.emptyState,
      params: {},
      schedule: SCHEDULE
    });
  } catch (e) {
    logger.error(`Error scheduling ${TASK_ID}, received ${e.message}`);
  }
}
function telemetryTaskRunner(logger, core, eventLogIndex, taskManagerIndex) {
  return ({
    taskInstance
  }) => {
    const state = taskInstance.state;
    const getEsClient = () => core.getStartServices().then(([{
      elasticsearch: {
        client
      }
    }]) => client.asInternalUser);
    const getAlertIndex = () => core.getStartServices().then(([coreStart]) => coreStart.savedObjects.getIndexForType(_saved_objects.RULE_SAVED_OBJECT_TYPE));
    const getSavedObjectClient = () => core.getStartServices().then(([coreStart]) => coreStart.savedObjects.createInternalRepository([_common.MAINTENANCE_WINDOW_SAVED_OBJECT_TYPE]));
    return {
      async run() {
        const esClient = await getEsClient();
        const alertIndex = await getAlertIndex();
        const savedObjectsClient = await getSavedObjectClient();
        return Promise.all([(0, _get_telemetry_from_kibana.getTotalCountAggregations)({
          esClient,
          alertIndex,
          logger
        }), (0, _get_telemetry_from_kibana.getTotalCountInUse)({
          esClient,
          alertIndex,
          logger
        }), (0, _get_telemetry_from_event_log.getExecutionsPerDayCount)({
          esClient,
          eventLogIndex,
          logger
        }), (0, _get_telemetry_from_event_log.getExecutionTimeoutsPerDayCount)({
          esClient,
          eventLogIndex,
          logger
        }), (0, _get_telemetry_from_task_manager.getFailedAndUnrecognizedTasksPerDay)({
          esClient,
          taskManagerIndex,
          logger
        }), (0, _get_telemetry_from_kibana.getMWTelemetry)({
          logger,
          savedObjectsClient
        }), (0, _get_telemetry_from_alerts.getTotalAlertsCountAggregations)({
          esClient,
          logger
        }), (0, _get_backfill_telemetry.getBackfillTelemetryPerDay)({
          esClient,
          eventLogIndex,
          logger
        })]).then(([totalCountAggregations, totalInUse, dailyExecutionCounts, dailyExecutionTimeoutCounts, dailyFailedAndUnrecognizedTasks, MWTelemetry, totalAlertsCountAggregations, dailyBackfillCounts]) => {
          const hasErrors = totalCountAggregations.hasErrors || totalInUse.hasErrors || dailyExecutionCounts.hasErrors || dailyExecutionTimeoutCounts.hasErrors || dailyFailedAndUnrecognizedTasks.hasErrors || MWTelemetry.hasErrors || totalAlertsCountAggregations.hasErrors || dailyBackfillCounts.hasErrors;
          const errorMessages = [totalCountAggregations.errorMessage, totalInUse.errorMessage, dailyExecutionCounts.errorMessage, dailyExecutionTimeoutCounts.errorMessage, dailyFailedAndUnrecognizedTasks.errorMessage, MWTelemetry.errorMessage, totalAlertsCountAggregations.errorMessage, dailyBackfillCounts.errorMessage].filter(message => message !== undefined);
          const updatedState = {
            has_errors: hasErrors,
            ...(errorMessages.length > 0 && {
              error_messages: errorMessages
            }),
            runs: (state.runs || 0) + 1,
            count_total: totalCountAggregations.count_total,
            count_by_type: totalCountAggregations.count_by_type,
            throttle_time: totalCountAggregations.throttle_time,
            schedule_time: totalCountAggregations.schedule_time,
            throttle_time_number_s: totalCountAggregations.throttle_time_number_s,
            schedule_time_number_s: totalCountAggregations.schedule_time_number_s,
            connectors_per_alert: totalCountAggregations.connectors_per_alert,
            count_active_by_type: totalInUse.countByType,
            count_active_total: totalInUse.countTotal,
            count_disabled_total: totalCountAggregations.count_total - totalInUse.countTotal,
            count_rules_by_execution_status: totalCountAggregations.count_rules_by_execution_status,
            count_rules_with_tags: totalCountAggregations.count_rules_with_tags,
            count_rules_by_notify_when: totalCountAggregations.count_rules_by_notify_when,
            count_rules_snoozed: totalCountAggregations.count_rules_snoozed,
            count_rules_muted: totalCountAggregations.count_rules_muted,
            count_rules_snoozed_by_type: totalCountAggregations.count_rules_snoozed_by_type,
            count_rules_muted_by_type: totalCountAggregations.count_rules_muted_by_type,
            count_rules_with_linked_dashboards: totalCountAggregations.count_rules_with_linked_dashboards,
            count_rules_with_investigation_guide: totalCountAggregations.count_rules_with_investigation_guide,
            count_mw_total: MWTelemetry.count_mw_total,
            count_mw_with_repeat_toggle_on: MWTelemetry.count_mw_with_repeat_toggle_on,
            count_mw_with_filter_alert_toggle_on: MWTelemetry.count_mw_with_filter_alert_toggle_on,
            count_rules_with_muted_alerts: totalCountAggregations.count_rules_with_muted_alerts,
            count_connector_types_by_consumers: totalCountAggregations.count_connector_types_by_consumers,
            count_rules_namespaces: totalInUse.countNamespaces,
            count_rules_executions_per_day: dailyExecutionCounts.countTotalRuleExecutions,
            count_rules_executions_by_type_per_day: dailyExecutionCounts.countRuleExecutionsByType,
            count_rules_executions_failured_per_day: dailyExecutionCounts.countTotalFailedExecutions,
            count_rules_executions_failured_by_reason_per_day: dailyExecutionCounts.countFailedExecutionsByReason,
            count_rules_executions_failured_by_reason_by_type_per_day: dailyExecutionCounts.countFailedExecutionsByReasonByType,
            count_rules_by_execution_status_per_day: dailyExecutionCounts.countRulesByExecutionStatus,
            count_rules_executions_timeouts_per_day: dailyExecutionTimeoutCounts.countExecutionTimeouts,
            count_rules_executions_timeouts_by_type_per_day: dailyExecutionTimeoutCounts.countExecutionTimeoutsByType,
            count_failed_and_unrecognized_rule_tasks_per_day: dailyFailedAndUnrecognizedTasks.countFailedAndUnrecognizedTasks,
            count_failed_and_unrecognized_rule_tasks_by_status_per_day: dailyFailedAndUnrecognizedTasks.countFailedAndUnrecognizedTasksByStatus,
            count_failed_and_unrecognized_rule_tasks_by_status_by_type_per_day: dailyFailedAndUnrecognizedTasks.countFailedAndUnrecognizedTasksByStatusByType,
            avg_execution_time_per_day: dailyExecutionCounts.avgExecutionTime,
            avg_execution_time_by_type_per_day: dailyExecutionCounts.avgExecutionTimeByType,
            avg_es_search_duration_per_day: dailyExecutionCounts.avgEsSearchDuration,
            avg_es_search_duration_by_type_per_day: dailyExecutionCounts.avgEsSearchDurationByType,
            avg_total_search_duration_per_day: dailyExecutionCounts.avgTotalSearchDuration,
            avg_total_search_duration_by_type_per_day: dailyExecutionCounts.avgTotalSearchDurationByType,
            percentile_num_generated_actions_per_day: dailyExecutionCounts.generatedActionsPercentiles,
            percentile_num_generated_actions_by_type_per_day: dailyExecutionCounts.generatedActionsPercentilesByType,
            percentile_num_alerts_per_day: dailyExecutionCounts.alertsPercentiles,
            percentile_num_alerts_by_type_per_day: dailyExecutionCounts.alertsPercentilesByType,
            count_alerts_total: totalAlertsCountAggregations.count_alerts_total,
            count_alerts_by_rule_type: totalAlertsCountAggregations.count_alerts_by_rule_type,
            count_backfill_executions: dailyBackfillCounts.countExecutions,
            count_backfills_by_execution_status_per_day: dailyBackfillCounts.countBackfillsByExecutionStatus,
            count_gaps: dailyBackfillCounts.countGaps,
            total_unfilled_gap_duration_ms: dailyBackfillCounts.totalUnfilledGapDurationMs,
            total_filled_gap_duration_ms: dailyBackfillCounts.totalFilledGapDurationMs,
            count_ignored_fields_by_rule_type: totalAlertsCountAggregations.count_ignored_fields_by_rule_type
          };
          return {
            state: updatedState,
            // Useful for setting a schedule for the old tasks that don't have one
            // or to update the schedule if ever the frequency changes in code
            schedule: SCHEDULE
          };
        }).catch(errMsg => {
          logger.warn(`Error executing alerting telemetry task: ${errMsg}`);
          return {
            state: _task_state.emptyState,
            // Useful for setting a schedule for the old tasks that don't have one
            // or to update the schedule if ever the frequency changes in code
            schedule: SCHEDULE
          };
        });
      }
    };
  };
}