"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SchedulerLoopState = void 0;
exports.processGapsForRules = processGapsForRules;
exports.processRuleBatches = processRuleBatches;
exports.registerGapAutoFillSchedulerTask = registerGapAutoFillSchedulerTask;
var _datemath = _interopRequireDefault(require("@kbn/datemath"));
var _find_gaps = require("../find_gaps");
var _process_gaps_batch = require("../../../application/gaps/methods/bulk_fill_gaps_by_rule_ids/process_gaps_batch");
var _types = require("../../../application/gaps/methods/bulk_fill_gaps_by_rule_ids/types");
var _constants = require("../../../../common/constants");
var _scheduler = require("../../../application/gaps/types/scheduler");
var _utils = require("./utils");
var _cleanup_stuck_in_progress_gaps = require("../update/cleanup_stuck_in_progress_gaps");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Circuit breaker to prevent infinite pagination loops when fetching gaps
const GAP_FETCH_MAX_ITERATIONS = 1000;
let SchedulerLoopState = exports.SchedulerLoopState = /*#__PURE__*/function (SchedulerLoopState) {
  SchedulerLoopState["COMPLETED"] = "completed";
  SchedulerLoopState["CAPACITY_EXHAUSTED"] = "capacity_exhausted";
  SchedulerLoopState["CANCELLED"] = "cancelled";
  return SchedulerLoopState;
}({});
async function processRuleBatches({
  abortController,
  gapsPerPage,
  gapFetchMaxIterations,
  logger,
  loggerMessage,
  logEvent,
  remainingBackfills,
  ruleIds,
  rulesBatchSize,
  rulesClient,
  rulesClientContext,
  sortOrder,
  startISO,
  endISO,
  taskInstanceId
}) {
  let aggregatedByRule = new Map();
  for (let startIdx = 0; startIdx < ruleIds.length; startIdx += rulesBatchSize) {
    if (remainingBackfills <= 0) {
      return {
        aggregatedByRule,
        state: SchedulerLoopState.CAPACITY_EXHAUSTED
      };
    }
    if ((0, _utils.isCancelled)(abortController)) {
      return {
        aggregatedByRule,
        state: SchedulerLoopState.CANCELLED
      };
    }
    const currentRuleIds = ruleIds.slice(startIdx, startIdx + rulesBatchSize);
    const {
      data: rules
    } = await rulesClient.find({
      options: {
        page: 1,
        perPage: currentRuleIds.length,
        filter: `alert.attributes.enabled:true AND (${currentRuleIds.map(id => `alert.id: ("alert:${id}")`).join(' OR ')})`
      }
    });
    const toProcessRuleIds = rules.map(rule => rule.id).slice(0, remainingBackfills);
    if (!toProcessRuleIds.length) {
      continue;
    }
    const gapsResult = await processGapsForRules({
      abortController,
      aggregatedByRule,
      endISO,
      gapsPerPage,
      gapFetchMaxIterations,
      logger,
      loggerMessage,
      logEvent,
      remainingBackfills,
      rulesClientContext,
      sortOrder,
      startISO,
      taskInstanceId,
      toProcessRuleIds
    });
    aggregatedByRule = gapsResult.aggregatedByRule;
    remainingBackfills = gapsResult.remainingBackfills;
    if (gapsResult.state !== SchedulerLoopState.COMPLETED) {
      return {
        aggregatedByRule,
        state: gapsResult.state
      };
    }
  }
  return {
    aggregatedByRule,
    state: SchedulerLoopState.COMPLETED
  };
}
async function processGapsForRules({
  abortController,
  aggregatedByRule,
  endISO,
  gapsPerPage,
  gapFetchMaxIterations,
  logger,
  loggerMessage,
  logEvent,
  remainingBackfills,
  rulesClientContext,
  sortOrder,
  startISO,
  taskInstanceId,
  toProcessRuleIds
}) {
  let aggregated = new Map(aggregatedByRule);
  let searchAfter;
  let pitId;
  let gapFetchIterationCount = 0;
  while (true) {
    if (gapFetchIterationCount >= gapFetchMaxIterations) {
      logger.debug(loggerMessage('Circuit breaker triggered: reached maximum number of gap fetch iterations'));
      break;
    }
    gapFetchIterationCount++;
    if ((0, _utils.isCancelled)(abortController)) {
      return {
        aggregatedByRule: aggregated,
        remainingBackfills,
        state: SchedulerLoopState.CANCELLED
      };
    }
    const {
      data: gapsPage,
      searchAfter: nextSearchAfter,
      pitId: nextPitId
    } = await (0, _find_gaps.findGapsSearchAfter)({
      eventLogClient: await rulesClientContext.getEventLogClient(),
      logger,
      params: {
        ruleIds: toProcessRuleIds,
        start: startISO,
        end: endISO,
        perPage: gapsPerPage,
        sortField: '@timestamp',
        sortOrder,
        statuses: [_constants.gapStatus.UNFILLED, _constants.gapStatus.PARTIALLY_FILLED],
        searchAfter,
        pitId,
        hasUnfilledIntervals: true
      }
    });
    pitId = nextPitId !== null && nextPitId !== void 0 ? nextPitId : pitId;
    searchAfter = nextSearchAfter;
    if (!gapsPage.length) {
      break;
    }
    const filteredGaps = await (0, _utils.filterGapsWithOverlappingBackfills)(gapsPage, rulesClientContext, message => logger.warn(loggerMessage(message)));
    if (filteredGaps.length) {
      const sortedGaps = filteredGaps.sort((a, b) => a.range.gte.getTime() - b.range.gte.getTime());
      const {
        results: chunkResults
      } = await (0, _process_gaps_batch.processGapsBatch)(rulesClientContext, {
        gapsBatch: sortedGaps,
        range: {
          start: startISO,
          end: endISO
        },
        initiator: _constants.backfillInitiator.SYSTEM,
        initiatorId: taskInstanceId
      });
      aggregated = addChunkResultsToAggregation(aggregated, chunkResults);
      const chunkScheduledCount = chunkResults.reduce((count, result) => result.status === _types.GapFillSchedulePerRuleStatus.SUCCESS ? count + 1 : count, 0);
      if (chunkScheduledCount > 0) {
        remainingBackfills = Math.max(remainingBackfills - chunkScheduledCount, 0);
        if (remainingBackfills <= 0) {
          return {
            aggregatedByRule: aggregated,
            remainingBackfills,
            state: SchedulerLoopState.CAPACITY_EXHAUSTED
          };
        }
      }
    }
    if (gapsPage.length < gapsPerPage) {
      break;
    }
  }
  return {
    aggregatedByRule: aggregated,
    remainingBackfills,
    state: SchedulerLoopState.COMPLETED
  };
}
function addChunkResultsToAggregation(aggregatedByRule, chunkResults) {
  const nextAggregated = new Map(aggregatedByRule);
  for (const chunk of chunkResults) {
    var _chunk$processedGaps, _existing$error;
    const existing = nextAggregated.get(chunk.ruleId);
    if (!existing) {
      nextAggregated.set(chunk.ruleId, {
        ruleId: chunk.ruleId,
        processedGaps: chunk.processedGaps,
        status: chunk.status,
        error: chunk.error
      });
      continue;
    }
    let combinedStatus = existing.status;
    if (chunk.status === _types.GapFillSchedulePerRuleStatus.ERROR) {
      combinedStatus = _types.GapFillSchedulePerRuleStatus.ERROR;
    }
    nextAggregated.set(chunk.ruleId, {
      ruleId: chunk.ruleId,
      processedGaps: existing.processedGaps + ((_chunk$processedGaps = chunk.processedGaps) !== null && _chunk$processedGaps !== void 0 ? _chunk$processedGaps : 0),
      status: combinedStatus,
      error: (_existing$error = existing.error) !== null && _existing$error !== void 0 ? _existing$error : chunk.error
    });
  }
  return nextAggregated;
}

/**
 * Gap Auto Fill Scheduler task
 *
 * This function registers the Gap Auto Fill Scheduler task. It is used to scan for rules that
 * have detection gaps and schedules backfills to fill those gaps. The scheduler:
 * - Loads its runtime configuration from a `gap_auto_fill_scheduler` saved object
 *   (referenced by `configId` in task params)
 * - Cleans up stuck in-progress gaps that don't have corresponding backfills
 * - Honors a global backfill capacity limit before scheduling and tracks the
 *   remaining capacity locally throughout the task run
 * - Processes rules in batches, prioritizing the rules with the oldest gaps first
 * - Uses PIT + search_after pagination to fetch gaps efficiently from event log
 * - Skips gaps that overlap with already scheduled/running backfills to prevent duplicates
 * - Aggregates per-rule results and logs a single summarized event at the end
 * - Supports cancellation and shutdown via `cancel()` and AbortController
 */

function registerGapAutoFillSchedulerTask({
  taskManager,
  logger,
  getRulesClientWithRequest,
  eventLogger,
  schedulerConfig
}) {
  var _schedulerConfig$time;
  taskManager.registerTaskDefinitions({
    [_scheduler.GAP_AUTO_FILL_SCHEDULER_TASK_TYPE]: {
      title: 'Gap Auto Fill Scheduler',
      timeout: (_schedulerConfig$time = schedulerConfig === null || schedulerConfig === void 0 ? void 0 : schedulerConfig.timeout) !== null && _schedulerConfig$time !== void 0 ? _schedulerConfig$time : _scheduler.DEFAULT_GAP_AUTO_FILL_SCHEDULER_TIMEOUT,
      createTaskRunner: ({
        taskInstance,
        fakeRequest,
        abortController
      }) => {
        return {
          async run() {
            const loggerMessage = message => `[gap-fill-auto-scheduler-task][${taskInstance.id}] ${message}`;
            const startTime = new Date();
            // Step 1: Initialization
            let rulesClient;
            let rulesClientContext;
            let config;
            let logEvent;
            try {
              const initResult = await (0, _utils.initRun)({
                fakeRequest,
                getRulesClientWithRequest,
                eventLogger,
                taskInstance,
                startTime
              });
              rulesClient = initResult.rulesClient;
              rulesClientContext = initResult.rulesClientContext;
              config = initResult.config;
              logEvent = initResult.logEvent;
            } catch (e) {
              const errMsg = e instanceof Error ? e.message : String(e);
              logger.error(loggerMessage(`initialization failed: ${errMsg}`));
              // There no point in retrying the task if it's not initialized.
              return {
                state: {},
                shouldDeleteTask: true
              };
            }
            try {
              var _dateMath$parse;
              const now = new Date();
              const startDate = (_dateMath$parse = _datemath.default.parse(config.gapFillRange)) === null || _dateMath$parse === void 0 ? void 0 : _dateMath$parse.toDate();
              if (!startDate) {
                throw new Error(`Invalid gapFillRange: ${config.gapFillRange}`);
              }
              const startISO = startDate.toISOString();
              const endISO = now.toISOString();

              // Cleanup stuck in-progress gaps
              try {
                const eventLogClient = await rulesClientContext.getEventLogClient();
                await (0, _cleanup_stuck_in_progress_gaps.cleanupStuckInProgressGaps)({
                  rulesClientContext,
                  eventLogClient,
                  eventLogger,
                  logger,
                  startDate
                });
              } catch (cleanupError) {
                const errMsg = cleanupError instanceof Error ? cleanupError.message : String(cleanupError);
                logger.warn(loggerMessage(`cleanup of stuck in-progress gaps failed: ${errMsg}`));
                // Continue with normal flow even if cleanup fails
              }

              // Step 2: Capacity pre-check
              const capacityCheckInitial = await (0, _utils.checkBackfillCapacity)({
                rulesClient,
                maxBackfills: config.maxBackfills,
                logMessage: message => logger.warn(loggerMessage(message)),
                initiatorId: taskInstance.id
              });
              if (!capacityCheckInitial.canSchedule) {
                await logEvent({
                  status: _scheduler.GAP_AUTO_FILL_STATUS.SKIPPED,
                  results: [],
                  message: `Skipped execution: gap auto-fill capacity limit reached. This task can schedule at most ${capacityCheckInitial.maxBackfills} gap backfills at a time, and existing backfills must finish before new ones can be scheduled.`
                });
                return {
                  state: {}
                };
              }

              // Step 3: Fetch rule IDs with gaps
              const remainingBackfills = capacityCheckInitial.remainingCapacity;
              // newest gap first
              const sortOrder = 'desc';
              const {
                ruleIds
              } = await rulesClient.getRuleIdsWithGaps({
                start: startISO,
                end: endISO,
                sortOrder,
                hasUnfilledIntervals: true,
                ruleTypes: config.ruleTypes
              });
              if (!ruleIds.length) {
                await logEvent({
                  status: _scheduler.GAP_AUTO_FILL_STATUS.SKIPPED,
                  results: [],
                  message: 'Skipped execution: no rules with gaps'
                });
                return {
                  state: {}
                };
              }

              // Step 4: Process rules in batches
              const gapFillsResult = await processRuleBatches({
                abortController,
                gapsPerPage: _scheduler.DEFAULT_GAPS_PER_PAGE,
                gapFetchMaxIterations: GAP_FETCH_MAX_ITERATIONS,
                logger,
                loggerMessage,
                logEvent,
                remainingBackfills,
                ruleIds,
                rulesBatchSize: _scheduler.DEFAULT_RULES_BATCH_SIZE,
                rulesClient,
                rulesClientContext,
                sortOrder,
                startISO,
                endISO,
                taskInstanceId: taskInstance.id
              });
              const aggregatedByRule = gapFillsResult.aggregatedByRule;
              const consolidated = (0, _utils.resultsFromMap)(aggregatedByRule);
              if (gapFillsResult.state === SchedulerLoopState.CAPACITY_EXHAUSTED) {
                await logEvent({
                  status: _scheduler.GAP_AUTO_FILL_STATUS.SUCCESS,
                  results: consolidated,
                  message: `Stopped early: gap auto-fill capacity limit reached. This task can schedule at most ${capacityCheckInitial.maxBackfills} gap backfills at a time, and existing backfills must finish before new ones can be scheduled. | ${(0, _utils.formatConsolidatedSummary)(consolidated)}`
                });
                return {
                  state: {}
                };
              }
              if (gapFillsResult.state === SchedulerLoopState.CANCELLED) {
                await logEvent({
                  status: _scheduler.GAP_AUTO_FILL_STATUS.SUCCESS,
                  results: consolidated,
                  message: `Gap Auto Fill Scheduler cancelled by timeout | Results: ${(0, _utils.formatConsolidatedSummary)(consolidated)}`
                });
                return {
                  state: {}
                };
              }

              // Step 5: Finalize and log results
              const {
                status: outcomeStatus,
                message: outcomeMessage
              } = (0, _utils.getGapAutoFillRunOutcome)(consolidated);
              const summary = consolidated.length ? ` | ${(0, _utils.formatConsolidatedSummary)(consolidated)}` : '';
              await logEvent({
                status: outcomeStatus,
                results: consolidated,
                message: `${outcomeMessage}${summary}`
              });
              return {
                state: {}
              };
            } catch (error) {
              await logEvent({
                status: _scheduler.GAP_AUTO_FILL_STATUS.ERROR,
                results: [],
                message: `Error during execution: ${error && error.message}`
              });
              logger.error(loggerMessage(`error: ${error && error.message}`));
              return {
                state: {}
              };
            }
          }
        };
      }
    }
  });
}