"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findGapsSearchAfter = exports.findGaps = void 0;
var _saved_objects = require("../../saved_objects");
var _transform_to_gap = require("./transforms/transform_to_gap");
var _build_gaps_filter = require("./build_gaps_filter");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const findGaps = async ({
  eventLogClient,
  logger,
  params
}) => {
  const {
    ruleId,
    start,
    end,
    page,
    perPage,
    statuses,
    sortField,
    sortOrder
  } = params;
  try {
    const filter = (0, _build_gaps_filter.buildGapsFilter)({
      start,
      end,
      statuses
    });
    const gapsResponse = await eventLogClient.findEventsBySavedObjectIds(_saved_objects.RULE_SAVED_OBJECT_TYPE, [ruleId], {
      filter,
      sort: [{
        sort_field: sortField !== null && sortField !== void 0 ? sortField : '@timestamp',
        sort_order: sortOrder !== null && sortOrder !== void 0 ? sortOrder : 'desc'
      }],
      page,
      per_page: perPage
    });
    return {
      total: gapsResponse.total,
      data: (0, _transform_to_gap.transformToGap)(gapsResponse),
      page: gapsResponse.page,
      perPage: gapsResponse.per_page
    };
  } catch (err) {
    logger.error(`Failed to find gaps for rule ${ruleId}: ${err.message}`);
    throw err;
  }
};
exports.findGaps = findGaps;
const FIND_GAPS_SEARCH_AFTER_MAX_RULES = 100;

/**
 * This function is used to find gaps using search after.
 * It's used when to be able process more than 10,000 gaps with stable sorting.
 */
const findGapsSearchAfter = async ({
  eventLogClient,
  logger,
  params
}) => {
  const {
    ruleIds,
    start,
    end,
    perPage,
    statuses,
    sortField,
    sortOrder,
    hasUnfilledIntervals,
    hasInProgressIntervals,
    hasFilledIntervals,
    updatedBefore
  } = params;
  if (ruleIds.length > FIND_GAPS_SEARCH_AFTER_MAX_RULES) {
    throw new Error(`ruleIds max size must be ${FIND_GAPS_SEARCH_AFTER_MAX_RULES}`);
  }
  try {
    const filter = (0, _build_gaps_filter.buildGapsFilter)({
      start,
      end,
      statuses,
      hasUnfilledIntervals,
      hasInProgressIntervals,
      hasFilledIntervals,
      updatedBefore
    });
    const gapsResponse = await eventLogClient.findEventsBySavedObjectIdsSearchAfter(_saved_objects.RULE_SAVED_OBJECT_TYPE, ruleIds, {
      filter,
      sort: [{
        sort_field: sortField !== null && sortField !== void 0 ? sortField : '@timestamp',
        sort_order: sortOrder !== null && sortOrder !== void 0 ? sortOrder : 'desc'
      }],
      per_page: perPage,
      pit_id: params === null || params === void 0 ? void 0 : params.pitId,
      search_after: params === null || params === void 0 ? void 0 : params.searchAfter
    });
    return {
      total: gapsResponse.total,
      data: (0, _transform_to_gap.transformToGap)(gapsResponse),
      searchAfter: gapsResponse.search_after,
      pitId: gapsResponse.pit_id
    };
  } catch (err) {
    logger.error(`Failed to find gaps with search after for rules ${ruleIds.join(', ')}: ${err.message}`);
    throw err;
  }
};
exports.findGapsSearchAfter = findGapsSearchAfter;