import {
  AGENT_CARD_PATH
} from "../../chunk-67JNQ6TZ.js";
import {
  A2AError,
  JsonRpcTransportHandler
} from "../../chunk-JA52GYRU.js";

// src/server/express/a2a_express_app.ts
import express from "express";
var A2AExpressApp = class {
  requestHandler;
  // Kept for getAgentCard
  jsonRpcTransportHandler;
  constructor(requestHandler) {
    this.requestHandler = requestHandler;
    this.jsonRpcTransportHandler = new JsonRpcTransportHandler(requestHandler);
  }
  /**
   * Adds A2A routes to an existing Express app.
   * @param app Optional existing Express app.
   * @param baseUrl The base URL for A2A endpoints (e.g., "/a2a/api").
   * @param middlewares Optional array of Express middlewares to apply to the A2A routes.
   * @param agentCardPath Optional custom path for the agent card endpoint (defaults to /.well-known/agent-card.json).
   * @returns The Express app with A2A routes.
   */
  setupRoutes(app, baseUrl = "", middlewares, agentCardPath = AGENT_CARD_PATH) {
    const router = express.Router();
    router.use(express.json(), ...middlewares ?? []);
    router.get(`/${agentCardPath}`, async (req, res) => {
      try {
        const agentCard = await this.requestHandler.getAgentCard();
        res.json(agentCard);
      } catch (error) {
        console.error("Error fetching agent card:", error);
        res.status(500).json({ error: "Failed to retrieve agent card" });
      }
    });
    router.post("/", async (req, res) => {
      try {
        const rpcResponseOrStream = await this.jsonRpcTransportHandler.handle(req.body);
        if (typeof rpcResponseOrStream?.[Symbol.asyncIterator] === "function") {
          const stream = rpcResponseOrStream;
          res.setHeader("Content-Type", "text/event-stream");
          res.setHeader("Cache-Control", "no-cache");
          res.setHeader("Connection", "keep-alive");
          res.flushHeaders();
          try {
            for await (const event of stream) {
              res.write(`id: ${(/* @__PURE__ */ new Date()).getTime()}
`);
              res.write(`data: ${JSON.stringify(event)}

`);
            }
          } catch (streamError) {
            console.error(`Error during SSE streaming (request ${req.body?.id}):`, streamError);
            const a2aError = streamError instanceof A2AError ? streamError : A2AError.internalError(streamError.message || "Streaming error.");
            const errorResponse = {
              jsonrpc: "2.0",
              id: req.body?.id || null,
              // Use original request ID if available
              error: a2aError.toJSONRPCError()
            };
            if (!res.headersSent) {
              res.status(500).json(errorResponse);
            } else {
              res.write(`id: ${(/* @__PURE__ */ new Date()).getTime()}
`);
              res.write(`event: error
`);
              res.write(`data: ${JSON.stringify(errorResponse)}

`);
            }
          } finally {
            if (!res.writableEnded) {
              res.end();
            }
          }
        } else {
          const rpcResponse = rpcResponseOrStream;
          res.status(200).json(rpcResponse);
        }
      } catch (error) {
        console.error("Unhandled error in A2AExpressApp POST handler:", error);
        const a2aError = error instanceof A2AError ? error : A2AError.internalError("General processing error.");
        const errorResponse = {
          jsonrpc: "2.0",
          id: req.body?.id || null,
          error: a2aError.toJSONRPCError()
        };
        if (!res.headersSent) {
          res.status(500).json(errorResponse);
        } else if (!res.writableEnded) {
          res.end();
        }
      }
    });
    app.use(baseUrl, router);
    return app;
  }
};
export {
  A2AExpressApp
};
