"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useImagePasteUpload = useImagePasteUpload;
var _react = require("react");
var _sharedUxFileContext = require("@kbn/shared-ux-file-context");
var _upload_state = require("@kbn/shared-ux-file-upload/src/upload_state");
var _translations = require("../translations");
var _constants = require("../constants");
var _files = require("../../../../common/files");
var _types = require("./types");
var _reducer = require("./reducer");
var _utils = require("./utils");
var _use_upload_start = require("./use_upload_start");
var _use_upload_complete = require("./use_upload_complete");
var _use_file_upload_api = require("./use_file_upload_api");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_STATE = {
  phase: _types.UploadPhase.IDLE
};
function useImagePasteUpload({
  editorRef,
  field,
  caseId,
  owner,
  fileKindId
}) {
  const {
    client
  } = (0, _sharedUxFileContext.useFilesContext)();
  const kind = client.getFileKind(fileKindId);
  const uploadState = (0, _react.useMemo)(() => (0, _upload_state.createUploadState)({
    client,
    fileKind: kind,
    allowRepeatedUploads: false
  }), [client, kind]);
  const [uiState, dispatch] = (0, _react.useReducer)(_reducer.reducer, DEFAULT_STATE);
  const textarea = (0, _utils.getTextarea)(editorRef);
  (0, _use_file_upload_api.useFileUploadApi)(uploadState, uiState, dispatch);
  const replacePlaceholder = (0, _react.useCallback)((file, placeholder) => {
    if (!textarea) return;
    const newText = textarea.value.replace(placeholder, (0, _utils.markdownImage)(file.fileJSON.name, client.getDownloadHref({
      id: file.id,
      fileKind: (0, _files.constructFileKindIdByOwner)(owner)
    }), file.fileJSON.extension));
    field.setValue(newText);
  }, [textarea, owner, client, field]);
  const pasteHandlerRef = (0, _react.useRef)();
  (0, _use_upload_start.useUploadStart)(uiState, dispatch, textarea, field);
  (0, _use_upload_complete.useUploadComplete)(uiState, replacePlaceholder, dispatch);

  /**
   * This hook handles paste events and input errors.
   * It calls the files API to begin uploading accepted image files.
   */
  (0, _react.useLayoutEffect)(() => {
    pasteHandlerRef.current = e => {
      var _e$clipboardData;
      const clipboardItems = (_e$clipboardData = e.clipboardData) === null || _e$clipboardData === void 0 ? void 0 : _e$clipboardData.items;
      if (!clipboardItems) return;
      const items = Array.from(clipboardItems).map(item => item === null || item === void 0 ? void 0 : item.getAsFile()).filter(item => !!item);

      // NOTE: In Firefox, there will always be only 1 or 0 items,
      // see: https://bugzilla.mozilla.org/show_bug.cgi?id=1699743
      if (items.length > 1) {
        dispatch({
          type: _types.ActionType.UPLOAD_ERROR,
          errors: [_translations.NO_SIMULTANEOUS_UPLOADS_MESSAGE]
        });
        e.preventDefault();
        return;
      }
      if (items.length === 0 || !items[0]) {
        // Not a file paste – clear any previous errors and allow default behaviour.
        if (uiState.phase === _types.UploadPhase.ERROR) {
          dispatch({
            type: _types.ActionType.RESET
          });
        }
        return;
      }
      const fileToUpload = items[0];
      if (!_constants.SUPPORTED_PASTE_MIME_TYPES.includes(fileToUpload.type)) {
        dispatch({
          type: _types.ActionType.UPLOAD_ERROR,
          errors: [_translations.UNSUPPORTED_MIME_TYPE_MESSAGE]
        });
        e.preventDefault();
        return;
      }

      // prevent the default paste behavior for supported image file pastes
      e.preventDefault();
      try {
        if (!uploadState.isUploading()) {
          uploadState.setFiles([fileToUpload]);
        }
      } catch (err) {
        dispatch({
          type: _types.ActionType.UPLOAD_ERROR,
          errors: [err]
        });
      }
      if ((0, _utils.canUpload)(uploadState, caseId)) {
        dispatch({
          type: _types.ActionType.RESET
        });
        uploadState.upload({
          caseIds: [caseId],
          owner
        });
      }
    };
  }, [caseId, owner, uiState.phase, uploadState]);

  // Attach the listener once for the lifetime of the textarea element.
  (0, _react.useLayoutEffect)(() => {
    if (!textarea) return;
    const delegatingListener = e => {
      var _pasteHandlerRef$curr;
      return (_pasteHandlerRef$curr = pasteHandlerRef.current) === null || _pasteHandlerRef$curr === void 0 ? void 0 : _pasteHandlerRef$curr.call(pasteHandlerRef, e);
    };
    textarea.addEventListener('paste', delegatingListener);
    return () => {
      textarea.removeEventListener('paste', delegatingListener);
    };
  }, [textarea]);
  return {
    isUploading: uiState.phase === _types.UploadPhase.UPLOADING,
    uploadState,
    errors: uiState.phase === _types.UploadPhase.ERROR ? uiState.errors : undefined
  };
}