/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.s3;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.NoSuchFileException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.common.blobstore.OperationPurpose;
import org.elasticsearch.core.IOUtils;
import org.elasticsearch.core.Strings;
import org.elasticsearch.repositories.blobstore.RequestedRangeNotSatisfiedException;
import org.elasticsearch.repositories.s3.AmazonS3Reference;
import org.elasticsearch.repositories.s3.S3BlobStore;
import org.elasticsearch.rest.RestStatus;
import software.amazon.awssdk.awscore.AwsRequest;
import software.amazon.awssdk.core.ResponseInputStream;
import software.amazon.awssdk.core.exception.SdkException;
import software.amazon.awssdk.core.exception.SdkServiceException;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.s3.model.GetObjectResponse;

class S3RetryingInputStream
extends InputStream {
    private static final Logger logger = LogManager.getLogger(S3RetryingInputStream.class);
    static final int MAX_SUPPRESSED_EXCEPTIONS = 10;
    private final OperationPurpose purpose;
    private final S3BlobStore blobStore;
    private final String blobKey;
    private final long start;
    private final long end;
    private final List<Exception> failures;
    private ResponseInputStream<GetObjectResponse> currentStream;
    private long currentStreamFirstOffset;
    private long currentStreamLastOffset;
    private int attempt = 1;
    private int failuresAfterMeaningfulProgress = 0;
    private long currentOffset;
    private boolean closed;
    private boolean eof;
    private boolean aborted = false;

    S3RetryingInputStream(OperationPurpose purpose, S3BlobStore blobStore, String blobKey) throws IOException {
        this(purpose, blobStore, blobKey, 0L, 0x7FFFFFFFFFFFFFFEL);
    }

    S3RetryingInputStream(OperationPurpose purpose, S3BlobStore blobStore, String blobKey, long start, long end) throws IOException {
        if (start < 0L) {
            throw new IllegalArgumentException("start must be non-negative");
        }
        if (end < start || end == Long.MAX_VALUE) {
            throw new IllegalArgumentException("end must be >= start and not Long.MAX_VALUE");
        }
        this.purpose = purpose;
        this.blobStore = blobStore;
        this.blobKey = blobKey;
        this.failures = new ArrayList<Exception>(10);
        this.start = start;
        this.end = end;
        int initialAttempt = this.attempt;
        this.openStreamWithRetry();
        this.maybeLogAndRecordMetricsForSuccess(initialAttempt, "open");
    }

    private void openStreamWithRetry() throws IOException {
        while (true) {
            try (AmazonS3Reference clientReference = this.blobStore.clientReference();){
                GetObjectRequest.Builder getObjectRequestBuilder = GetObjectRequest.builder().bucket(this.blobStore.bucket()).key(this.blobKey);
                S3BlobStore.configureRequestForMetrics((AwsRequest.Builder)getObjectRequestBuilder, this.blobStore, S3BlobStore.Operation.GET_OBJECT, this.purpose);
                if (this.currentOffset > 0L || this.start > 0L || this.end < 0x7FFFFFFFFFFFFFFEL) {
                    assert (this.start + this.currentOffset <= this.end) : "requesting beyond end, start = " + this.start + " offset=" + this.currentOffset + " end=" + this.end;
                    getObjectRequestBuilder.range("bytes=" + Math.addExact(this.start, this.currentOffset) + "-" + this.end);
                }
                this.currentStreamFirstOffset = Math.addExact(this.start, this.currentOffset);
                GetObjectRequest getObjectRequest = (GetObjectRequest)getObjectRequestBuilder.build();
                ResponseInputStream getObjectResponse = clientReference.client().getObject(getObjectRequest);
                this.currentStreamLastOffset = Math.addExact(this.currentStreamFirstOffset, this.getStreamLength((GetObjectResponse)getObjectResponse.response()));
                this.currentStream = getObjectResponse;
                return;
            }
            catch (SdkException e) {
                if (e instanceof SdkServiceException) {
                    SdkServiceException sdkServiceException = (SdkServiceException)((Object)e);
                    if (sdkServiceException.statusCode() == RestStatus.NOT_FOUND.getStatus()) {
                        throw this.addSuppressedExceptions(new NoSuchFileException("Blob object [" + this.blobKey + "] not found: " + sdkServiceException.getMessage()));
                    }
                    if (sdkServiceException.statusCode() == RestStatus.REQUESTED_RANGE_NOT_SATISFIED.getStatus()) {
                        throw this.addSuppressedExceptions(new RequestedRangeNotSatisfiedException(this.blobKey, this.currentStreamFirstOffset, this.end < 0x7FFFFFFFFFFFFFFEL ? this.end - this.currentStreamFirstOffset + 1L : this.end, (Throwable)sdkServiceException));
                    }
                }
                if (this.attempt == 1) {
                    this.blobStore.getS3RepositoriesMetrics().retryStartedCounter().incrementBy(1L, this.metricAttributes("open"));
                }
                long delayInMillis = this.maybeLogAndComputeRetryDelay("opening", e);
                this.delayBeforeRetry(delayInMillis);
                continue;
            }
            break;
        }
    }

    private long getStreamLength(GetObjectResponse getObjectResponse) {
        try {
            return this.tryGetStreamLength(getObjectResponse);
        }
        catch (Exception e) {
            assert (false) : e;
            return 0x7FFFFFFFFFFFFFFEL;
        }
    }

    long tryGetStreamLength(GetObjectResponse getObjectResponse) {
        String rangeString = getObjectResponse.contentRange();
        if (rangeString != null) {
            if (!rangeString.startsWith("bytes ")) {
                throw new IllegalArgumentException("unexpected Content-range header [" + rangeString + "], should have started with [bytes ]");
            }
            int hyphenPos = rangeString.indexOf(45);
            if (hyphenPos == -1) {
                throw new IllegalArgumentException("could not parse Content-range header [" + rangeString + "], missing hyphen");
            }
            int slashPos = rangeString.indexOf(47);
            if (slashPos == -1) {
                throw new IllegalArgumentException("could not parse Content-range header [" + rangeString + "], missing slash");
            }
            long rangeStart = Long.parseLong(rangeString, "bytes ".length(), hyphenPos, 10);
            long rangeEnd = Long.parseLong(rangeString, hyphenPos + 1, slashPos, 10);
            if (rangeEnd < rangeStart) {
                throw new IllegalArgumentException("invalid Content-range header [" + rangeString + "]");
            }
            if (rangeStart != this.start + this.currentOffset) {
                throw new IllegalArgumentException("unexpected Content-range header [" + rangeString + "], should have started at " + (this.start + this.currentOffset));
            }
            if (rangeEnd > this.end) {
                throw new IllegalArgumentException("unexpected Content-range header [" + rangeString + "], should have ended no later than " + this.end);
            }
            return rangeEnd - rangeStart + 1L;
        }
        return getObjectResponse.contentLength();
    }

    @Override
    public int read() throws IOException {
        this.ensureOpen();
        int initialAttempt = this.attempt;
        while (true) {
            try {
                int result = this.currentStream.read();
                if (result == -1) {
                    this.eof = true;
                } else {
                    ++this.currentOffset;
                }
                this.maybeLogAndRecordMetricsForSuccess(initialAttempt, "read");
                return result;
            }
            catch (IOException e) {
                if (this.attempt == initialAttempt) {
                    this.blobStore.getS3RepositoriesMetrics().retryStartedCounter().incrementBy(1L, this.metricAttributes("read"));
                }
                this.reopenStreamOrFail(e);
                continue;
            }
            break;
        }
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        this.ensureOpen();
        int initialAttempt = this.attempt;
        while (true) {
            try {
                int bytesRead = this.currentStream.read(b, off, len);
                if (bytesRead == -1) {
                    this.eof = true;
                } else {
                    this.currentOffset += (long)bytesRead;
                }
                this.maybeLogAndRecordMetricsForSuccess(initialAttempt, "read");
                return bytesRead;
            }
            catch (IOException e) {
                if (this.attempt == initialAttempt) {
                    this.blobStore.getS3RepositoriesMetrics().retryStartedCounter().incrementBy(1L, this.metricAttributes("read"));
                }
                this.reopenStreamOrFail(e);
                continue;
            }
            break;
        }
    }

    private void ensureOpen() {
        if (this.closed) {
            assert (false) : "using S3RetryingInputStream after close";
            throw new IllegalStateException("using S3RetryingInputStream after close");
        }
    }

    private void reopenStreamOrFail(IOException e) throws IOException {
        long meaningfulProgressSize = Math.max(1L, this.blobStore.bufferSizeInBytes() / 100L);
        if (this.currentStreamProgress() >= meaningfulProgressSize) {
            ++this.failuresAfterMeaningfulProgress;
        }
        long delayInMillis = this.maybeLogAndComputeRetryDelay("reading", e);
        this.maybeAbort(this.currentStream);
        IOUtils.closeWhileHandlingException(this.currentStream);
        this.delayBeforeRetry(delayInMillis);
        this.openStreamWithRetry();
    }

    private <T extends Exception> long maybeLogAndComputeRetryDelay(String action, T e) throws T {
        if (!this.shouldRetry(this.attempt)) {
            T finalException = this.addSuppressedExceptions(e);
            this.logForFailure(action, finalException);
            throw finalException;
        }
        this.logForRetry(Integer.bitCount(this.attempt) == 1 ? Level.INFO : Level.DEBUG, action, e);
        if (this.failures.size() < 10) {
            this.failures.add(e);
        }
        long delayInMillis = this.getRetryDelayInMillis();
        ++this.attempt;
        return delayInMillis;
    }

    private void logForFailure(String action, Exception e) {
        logger.warn(() -> Strings.format((String)"failed %s [%s/%s] at offset [%s] with purpose [%s]", (Object[])new Object[]{action, this.blobStore.bucket(), this.blobKey, this.start + this.currentOffset, this.purpose.getKey()}), (Throwable)e);
    }

    private void logForRetry(Level level, String action, Exception e) {
        logger.log(level, () -> Strings.format((String)"failed %s [%s/%s] at offset [%s] with purpose [%s]; this was attempt [%s] to read this blob which yielded [%s] bytes; in total [%s] of the attempts to read this blob have made meaningful progress and do not count towards the maximum number of retries; the maximum number of read attempts which do not make meaningful progress is [%s]", (Object[])new Object[]{action, this.blobStore.bucket(), this.blobKey, this.start + this.currentOffset, this.purpose.getKey(), this.attempt, this.currentStreamProgress(), this.failuresAfterMeaningfulProgress, this.maxRetriesForNoMeaningfulProgress()}), (Throwable)e);
    }

    private void maybeLogAndRecordMetricsForSuccess(int initialAttempt, String action) {
        if (this.attempt > initialAttempt) {
            int numberOfRetries = this.attempt - initialAttempt;
            logger.info("successfully {} input stream for [{}/{}] with purpose [{}] after [{}] retries", (Object)action, (Object)this.blobStore.bucket(), (Object)this.blobKey, (Object)this.purpose.getKey(), (Object)numberOfRetries);
            Map<String, Object> attributes = this.metricAttributes(action);
            this.blobStore.getS3RepositoriesMetrics().retryCompletedCounter().incrementBy(1L, attributes);
            this.blobStore.getS3RepositoriesMetrics().retryHistogram().record((long)numberOfRetries, attributes);
        }
    }

    private long currentStreamProgress() {
        return Math.subtractExact(Math.addExact(this.start, this.currentOffset), this.currentStreamFirstOffset);
    }

    private boolean shouldRetry(int attempt) {
        if (this.purpose == OperationPurpose.REPOSITORY_ANALYSIS) {
            return false;
        }
        if (this.purpose == OperationPurpose.INDICES) {
            return true;
        }
        int maxAttempts = this.blobStore.getMaxRetries() + 1;
        return attempt < maxAttempts + this.failuresAfterMeaningfulProgress;
    }

    private int maxRetriesForNoMeaningfulProgress() {
        return this.purpose == OperationPurpose.INDICES ? Integer.MAX_VALUE : this.blobStore.getMaxRetries() + 1;
    }

    private void delayBeforeRetry(long delayInMillis) {
        try {
            assert (this.shouldRetry(this.attempt - 1)) : "should not have retried";
            Thread.sleep(delayInMillis);
        }
        catch (InterruptedException e) {
            logger.info("s3 input stream delay interrupted", (Throwable)e);
            Thread.currentThread().interrupt();
        }
    }

    protected long getRetryDelayInMillis() {
        return 10L << Math.min(this.attempt - 1, 10);
    }

    private Map<String, Object> metricAttributes(String action) {
        return Map.of("repo_type", "s3", "repo_name", this.blobStore.getRepositoryMetadata().name(), "operation", S3BlobStore.Operation.GET_OBJECT.getKey(), "purpose", this.purpose.getKey(), "action", action);
    }

    @Override
    public void close() throws IOException {
        this.maybeAbort(this.currentStream);
        try {
            this.currentStream.close();
        }
        finally {
            this.closed = true;
        }
    }

    private void maybeAbort(ResponseInputStream<?> stream) {
        if (this.isEof()) {
            return;
        }
        try {
            if (this.start + this.currentOffset < this.currentStreamLastOffset) {
                stream.abort();
                this.aborted = true;
            }
        }
        catch (Exception e) {
            logger.warn("Failed to abort stream before closing", (Throwable)e);
        }
    }

    @Override
    public long skip(long n) throws IOException {
        return super.skip(n);
    }

    @Override
    public void reset() {
        throw new UnsupportedOperationException("S3RetryingInputStream does not support seeking");
    }

    private <T extends Exception> T addSuppressedExceptions(T e) {
        for (Exception failure : this.failures) {
            e.addSuppressed(failure);
        }
        return e;
    }

    boolean isEof() {
        return this.eof || this.start + this.currentOffset == this.currentStreamLastOffset;
    }

    boolean isAborted() {
        return this.aborted;
    }
}

