"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.HealthDiagnosticServiceImpl = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _configSchema = require("@kbn/config-schema");
var _rxjs = require("rxjs");
var _lodash = require("lodash");
var _health_diagnostic_utils = require("./health_diagnostic_utils");
var _health_diagnostic_circuit_breakers = require("./health_diagnostic_circuit_breakers.types");
var _health_diagnostic_receiver = require("./health_diagnostic_receiver");
var _events = require("../event_based/events");
var _artifact = require("../artifact");
var _helpers = require("../helpers");
var _configuration = require("../configuration");
var _rss_growth_circuit_breaker = require("./circuit_breakers/rss_growth_circuit_breaker");
var _timeout_circuit_breaker = require("./circuit_breakers/timeout_circuit_breaker");
var _event_loop_utilization_circuit_breaker = require("./circuit_breakers/event_loop_utilization_circuit_breaker");
var _event_loop_delay_circuit_breaker = require("./circuit_breakers/event_loop_delay_circuit_breaker");
var _elastic_search_circuit_breaker = require("./circuit_breakers/elastic_search_circuit_breaker");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TASK_TYPE = 'security:health-diagnostic';
const TASK_ID = `${TASK_TYPE}:1.0.0`;
const INTERVAL = '1h';
const TIMEOUT = '10m';
const QUERY_ARTIFACT_ID = 'health-diagnostic-queries-v1';
class HealthDiagnosticServiceImpl {
  constructor(logger) {
    (0, _defineProperty2.default)(this, "salt", 'c2a5d101-d0ef-49cc-871e-6ee55f9546f8');
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "queryExecutor", void 0);
    (0, _defineProperty2.default)(this, "analytics", void 0);
    (0, _defineProperty2.default)(this, "_esClient", void 0);
    const mdc = {
      task_id: TASK_ID,
      task_type: TASK_TYPE
    };
    this.logger = (0, _helpers.newTelemetryLogger)(logger.get('health-diagnostic'), mdc);
  }
  setup(setup) {
    this.logger.debug('Setting up health diagnostic service');
    this.registerTask(setup.taskManager);
  }
  async start(start) {
    this.logger.debug('Starting health diagnostic service');
    this.queryExecutor = new _health_diagnostic_receiver.CircuitBreakingQueryExecutorImpl(start.esClient, this.logger);
    this.analytics = start.analytics;
    this._esClient = start.esClient;
    await this.scheduleTask(start.taskManager);
  }
  async runHealthDiagnosticQueries(lastExecutionByQuery) {
    this.logger.debug('Running health diagnostic task');
    const queriesToRun = await this.getRunnableHealthQueries(lastExecutionByQuery, new Date());
    const statistics = [];
    if (this.queryExecutor === undefined) {
      this.logger.warn('CircuitBreakingQueryExecutor service is not started');
      return statistics;
    }
    this.logger.debug('About to run health diagnostic queries', {
      queriesToRun: queriesToRun.length
    });
    for (const query of queriesToRun) {
      const now = new Date();
      const circuitBreakers = this.buildCircuitBreakers();
      const options = {
        query,
        circuitBreakers
      };
      const query$ = this.queryExecutor.search(options);
      const stats = await new Promise(resolve => {
        const queryStats = (0, _health_diagnostic_utils.emptyStat)(query.name, now);
        let currentPage = 0;
        query$.pipe(
        // cap the result set to the max number of documents
        (0, _rxjs.take)(_configuration.telemetryConfiguration.health_diagnostic_config.query.maxDocuments),
        // get the fields names, only once (assume all docs have the same structure)
        (0, _rxjs.tap)(doc => {
          if (queryStats.fieldNames.length === 0) {
            queryStats.fieldNames = (0, _health_diagnostic_utils.fieldNames)(doc);
          }
        }),
        // publish N documents in the same EBT
        (0, _rxjs.bufferCount)(_configuration.telemetryConfiguration.health_diagnostic_config.query.bufferSize),
        // emit empty array if no items were buffered (ensures EBT is always sent)
        (0, _rxjs.defaultIfEmpty)([]),
        // apply filterlist
        (0, _rxjs.mergeMap)(result => (0, _rxjs.from)((0, _health_diagnostic_utils.applyFilterlist)(result, query.filterlist, this.salt, query, _configuration.telemetryConfiguration.encryption_public_keys)))).subscribe({
          next: data => {
            this.logger.debug('Sending query result EBT', {
              queryName: query.name,
              traceId: queryStats.traceId
            });
            this.reportEBT(_events.TELEMETRY_HEALTH_DIAGNOSTIC_QUERY_RESULT_EVENT, {
              name: query.name,
              queryId: query.id,
              traceId: queryStats.traceId,
              page: currentPage++,
              data
            });
            queryStats.numDocs += data.length;
          },
          error: error => {
            const failure = {
              message: error.message,
              reason: error instanceof _health_diagnostic_circuit_breakers.ValidationError ? error.result : undefined
            };
            this.logger.error('Error running query', (0, _helpers.withErrorMessage)(error));
            resolve({
              ...queryStats,
              failure,
              finished: new Date().toISOString(),
              circuitBreakers: this.circuitBreakersStats(circuitBreakers),
              passed: false
            });
          },
          complete: () => {
            resolve({
              ...queryStats,
              finished: new Date().toISOString(),
              circuitBreakers: this.circuitBreakersStats(circuitBreakers),
              passed: true
            });
          }
        });
      });
      this.logger.debug('Query executed. Sending query stats EBT', {
        queryName: query.name,
        traceId: stats.traceId,
        statistics: stats
      });
      this.reportEBT(_events.TELEMETRY_HEALTH_DIAGNOSTIC_QUERY_STATS_EVENT, stats);
      statistics.push(stats);
    }
    this.logger.debug('Finished running health diagnostic task', {
      statistics
    });
    return statistics;
  }
  circuitBreakersStats(circuitBreakers) {
    return circuitBreakers.reduce((acc, cb) => {
      acc[cb.constructor.name] = cb.stats();
      return acc;
    }, {});
  }
  registerTask(taskManager) {
    this.logger.debug('About to register task');
    taskManager.registerTaskDefinitions({
      [TASK_TYPE]: {
        title: 'Security Solution - Health Diagnostic Task',
        description: 'This task periodically collects health diagnostic information.',
        timeout: TIMEOUT,
        maxAttempts: 1,
        stateSchemaByVersion: {
          1: {
            up: state => ({
              lastExecutionByQuery: state.lastExecutionByQuery || {}
            }),
            schema: _configSchema.schema.object({
              lastExecutionByQuery: _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.number())
            })
          }
        },
        createTaskRunner: ({
          taskInstance
        }) => {
          return {
            run: async () => {
              const {
                state
              } = taskInstance;
              const stats = await this.runHealthDiagnosticQueries((0, _lodash.cloneDeep)(state.lastExecutionByQuery));
              const lastExecutionByQuery = stats.reduce((acc, stat) => {
                acc[stat.name] = new Date(stat.finished).getTime();
                return acc;
              }, {});
              return {
                state: {
                  lastExecutionByQuery: {
                    ...state.lastExecutionByQuery,
                    ...lastExecutionByQuery
                  }
                }
              };
            },
            cancel: async () => {
              var _this$logger;
              (_this$logger = this.logger) === null || _this$logger === void 0 ? void 0 : _this$logger.warn('Task timed out');
            }
          };
        }
      }
    });
  }
  async scheduleTask(taskManager) {
    this.logger.info('About to schedule task');
    await taskManager.ensureScheduled({
      id: TASK_ID,
      taskType: TASK_TYPE,
      schedule: {
        interval: INTERVAL
      },
      params: {},
      state: {
        lastExecutionByQuery: {}
      },
      scope: ['securitySolution']
    });
    this.logger.info('Task scheduled');
  }
  buildCircuitBreakers() {
    const config = _configuration.telemetryConfiguration.health_diagnostic_config;
    return [new _rss_growth_circuit_breaker.RssGrowthCircuitBreaker(config.rssGrowthCircuitBreaker), new _timeout_circuit_breaker.TimeoutCircuitBreaker(config.timeoutCircuitBreaker), new _event_loop_utilization_circuit_breaker.EventLoopUtilizationCircuitBreaker(config.eventLoopUtilizationCircuitBreaker), new _event_loop_delay_circuit_breaker.EventLoopDelayCircuitBreaker(config.eventLoopDelayCircuitBreaker), new _elastic_search_circuit_breaker.ElasticsearchCircuitBreaker(config.elasticsearchCircuitBreaker, this.esClient())];
  }
  esClient() {
    if (this._esClient === undefined || this._esClient === null) {
      throw Error('elasticsearch client is unavailable');
    }
    return this._esClient;
  }
  reportEBT(eventTypeOpts, eventData) {
    if (!this.analytics) {
      throw Error('analytics is unavailable');
    }
    try {
      this.analytics.reportEvent(eventTypeOpts.eventType, eventData);
    } catch (error) {
      this.logger.warn('Error sending EBT', (0, _helpers.withErrorMessage)(error));
    }
  }
  async getRunnableHealthQueries(lastExecutionByQuery, now) {
    const healthQueries = await this.healthQueries();
    return healthQueries.filter(query => {
      try {
        var _lastExecutionByQuery;
        const {
          name,
          scheduleCron,
          enabled = false
        } = query;
        const lastExecutedAt = new Date((_lastExecutionByQuery = lastExecutionByQuery[name]) !== null && _lastExecutionByQuery !== void 0 ? _lastExecutionByQuery : 0);
        return enabled && (0, _health_diagnostic_utils.shouldExecute)(lastExecutedAt, now, scheduleCron);
      } catch (error) {
        this.logger.warn('Error processing health query', (0, _helpers.withErrorMessage)(error, {
          name: query.name
        }));
        return false;
      }
    });
  }
  async healthQueries() {
    try {
      const artifact = await _artifact.artifactService.getArtifact(QUERY_ARTIFACT_ID);
      return (0, _health_diagnostic_utils.parseDiagnosticQueries)(artifact.data);
    } catch (error) {
      this.logger.warn('Error getting health diagnostic queries', (0, _helpers.withErrorMessage)(error));
      return [];
    }
  }
}
exports.HealthDiagnosticServiceImpl = HealthDiagnosticServiceImpl;