"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createPrivilegeMonitoringIndicesRoute = void 0;
var _utils = require("@kbn/lists-plugin/server/routes/utils");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _zodHelpers = require("@kbn/zod-helpers");
var _entity_analytics = require("../../../../../common/api/entity_analytics");
var _constants = require("../../../../../common/constants");
var _assert_advanced_setting_enabled = require("../../utils/assert_advanced_setting_enabled");
var _data_sources_service = require("../data_sources/data_sources_service");
var _saved_object = require("../auth/saved_object");
var _saved_objects = require("../saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createPrivilegeMonitoringIndicesRoute = (router, logger) => {
  router.versioned.put({
    access: 'public',
    path: _constants.PRIVMON_INDICES_URL,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution', `${_constants.APP_ID}-entity-analytics`]
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {
      request: {
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_entity_analytics.CreatePrivilegesImportIndexRequestBody)
      }
    }
  }, async (context, request, response) => {
    const secSol = await context.securitySolution;
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    const indexName = request.body.name;
    const indexMode = request.body.mode;
    await (0, _assert_advanced_setting_enabled.assertAdvancedSettingsEnabled)(await context.core, _constants.ENABLE_PRIVILEGED_USER_MONITORING_SETTING);
    const dataClient = secSol.getPrivilegeMonitoringDataClient();
    const config = secSol.getConfig();
    const maxUsersAllowed = config.entityAnalytics.monitoring.privileges.users.maxPrivilegedUsersAllowed;
    const soClient = dataClient.getScopedSoClient(request, {
      includedHiddenTypes: [_saved_object.PrivilegeMonitoringApiKeyType.name, _saved_objects.monitoringEntitySourceType.name]
    });
    const dataSourcesService = (0, _data_sources_service.createDataSourcesService)(dataClient, soClient, maxUsersAllowed);
    try {
      await dataSourcesService.createImportIndex(indexName, indexMode);
      return response.ok();
    } catch (e) {
      const error = (0, _securitysolutionEsUtils.transformError)(e);
      logger.error(`Error creating privilege monitoring indices: ${error.message}`);
      return siemResponse.error({
        statusCode: error.statusCode,
        body: error.message
      });
    }
  });
};
exports.createPrivilegeMonitoringIndicesRoute = createPrivilegeMonitoringIndicesRoute;