"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.multiLineStringDiffAlgorithm = void 0;
var _nodeDiff = require("node-diff3");
var _utility_types = require("../../../../../../../../common/utility_types");
var _prebuilt_rules = require("../../../../../../../../common/api/detection_engine/prebuilt_rules");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Diff algorithm used for string fields that contain multiple lines
 */
const multiLineStringDiffAlgorithm = (versions, isRuleCustomized) => {
  const {
    base_version: baseVersion,
    current_version: currentVersion,
    target_version: targetVersion
  } = versions;
  const diffOutcome = (0, _prebuilt_rules.determineDiffOutcome)(baseVersion, currentVersion, targetVersion);
  const valueCanUpdate = (0, _prebuilt_rules.determineIfValueCanUpdate)(diffOutcome);
  const hasBaseVersion = baseVersion !== _prebuilt_rules.MissingVersion;
  const {
    mergeOutcome,
    conflict,
    mergedVersion
  } = mergeVersions({
    baseVersion: hasBaseVersion ? baseVersion : undefined,
    currentVersion,
    targetVersion,
    diffOutcome,
    isRuleCustomized
  });
  return {
    has_base_version: hasBaseVersion,
    base_version: hasBaseVersion ? baseVersion : undefined,
    current_version: currentVersion,
    target_version: targetVersion,
    merged_version: mergedVersion,
    merge_outcome: mergeOutcome,
    diff_outcome: diffOutcome,
    conflict,
    has_update: valueCanUpdate
  };
};
exports.multiLineStringDiffAlgorithm = multiLineStringDiffAlgorithm;
const mergeVersions = ({
  baseVersion,
  currentVersion,
  targetVersion,
  diffOutcome,
  isRuleCustomized
}) => {
  switch (diffOutcome) {
    case _prebuilt_rules.ThreeWayDiffOutcome.StockValueNoUpdate:
    case _prebuilt_rules.ThreeWayDiffOutcome.CustomizedValueNoUpdate:
    case _prebuilt_rules.ThreeWayDiffOutcome.CustomizedValueSameUpdate:
      return {
        conflict: _prebuilt_rules.ThreeWayDiffConflict.NONE,
        mergedVersion: currentVersion,
        mergeOutcome: _prebuilt_rules.ThreeWayMergeOutcome.Current
      };
    case _prebuilt_rules.ThreeWayDiffOutcome.StockValueCanUpdate:
      {
        return {
          conflict: _prebuilt_rules.ThreeWayDiffConflict.NONE,
          mergedVersion: targetVersion,
          mergeOutcome: _prebuilt_rules.ThreeWayMergeOutcome.Target
        };
      }
    case _prebuilt_rules.ThreeWayDiffOutcome.CustomizedValueCanUpdate:
      {
        // TS does not realize that in ABC scenario, baseVersion cannot be missing
        // Missing baseVersion scenarios were handled as -AA and -AB.
        const mergedVersion = (0, _nodeDiff.merge)(currentVersion, baseVersion !== null && baseVersion !== void 0 ? baseVersion : '', targetVersion, {
          stringSeparator: /(\r\n|\n|\r)/g // Separates strings by new lines
        });
        return mergedVersion.conflict ? {
          conflict: _prebuilt_rules.ThreeWayDiffConflict.NON_SOLVABLE,
          mergedVersion: currentVersion,
          mergeOutcome: _prebuilt_rules.ThreeWayMergeOutcome.Current
        } : {
          conflict: _prebuilt_rules.ThreeWayDiffConflict.SOLVABLE,
          mergedVersion: mergedVersion.result.join(''),
          mergeOutcome: _prebuilt_rules.ThreeWayMergeOutcome.Merged
        };
      }

    // Missing base versions always return target version
    // Scenario -AA is treated as AAA
    // https://github.com/elastic/kibana/issues/210358#issuecomment-2654492854
    case _prebuilt_rules.ThreeWayDiffOutcome.MissingBaseNoUpdate:
      {
        return {
          conflict: _prebuilt_rules.ThreeWayDiffConflict.NONE,
          mergedVersion: targetVersion,
          mergeOutcome: _prebuilt_rules.ThreeWayMergeOutcome.Target
        };
      }

    // Missing base versions always return target version
    // If the rule is customized, we return a SOLVABLE conflict
    // Since multi-line string fields are mergeable, we would typically return a merged value
    // as per https://github.com/elastic/kibana/pull/211862, but with no base version we cannot
    // complete a full diff merge and so just return the target version
    case _prebuilt_rules.ThreeWayDiffOutcome.MissingBaseCanUpdate:
      {
        return {
          conflict: isRuleCustomized ? _prebuilt_rules.ThreeWayDiffConflict.SOLVABLE : _prebuilt_rules.ThreeWayDiffConflict.NONE,
          mergedVersion: targetVersion,
          mergeOutcome: _prebuilt_rules.ThreeWayMergeOutcome.Target
        };
      }
    default:
      return (0, _utility_types.assertUnreachable)(diffOutcome);
  }
};