"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useCanAssignArtifactPerPolicy = void 0;
var _react = require("react");
var _use_license = require("../../../common/hooks/use_license");
var _artifacts = require("../../../../common/endpoint/service/artifacts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Calculates if by-policy assignments can be made to an artifact.
 *
 * Per-Policy assignment is a Platinum+ licensed feature only, but the component can
 * be displayed in down-grade conditions: meaning - when user downgrades the license,
 * we will still allow the component to be displayed in the UI so that user has the
 * ability to set the artifact to `global`.
 */
const useCanAssignArtifactPerPolicy = (item, mode, hasItemBeenUpdated, licenseType = 'platinumPlus') => {
  const license = (0, _use_license.useLicense)();
  const hasRequiredLicense = licenseType === 'platinumPlus' ? license.isPlatinumPlus() : license.isEnterprise();
  const isGlobal = (0, _react.useMemo)(() => (0, _artifacts.isArtifactGlobal)(item), [item]);
  const [wasByPolicy, setWasByPolicy] = (0, _react.useState)((0, _artifacts.isArtifactByPolicy)(item));
  (0, _react.useEffect)(() => {
    if (!hasItemBeenUpdated && item.tags) {
      setWasByPolicy(!(0, _artifacts.isArtifactGlobal)({
        tags: item.tags
      }));
    }
  }, [item.tags, hasItemBeenUpdated]);
  return (0, _react.useMemo)(() => {
    if (hasRequiredLicense) {
      return true;
    }
    if (mode !== 'edit') {
      return false;
    }
    if (!isGlobal) {
      return true;
    }
    return wasByPolicy && hasItemBeenUpdated;
  }, [mode, isGlobal, hasItemBeenUpdated, hasRequiredLicense, wasByPolicy]);
};
exports.useCanAssignArtifactPerPolicy = useCanAssignArtifactPerPolicy;