"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateToolId = exports.toolIdRegexp = exports.toolIdMaxLength = exports.isReservedToolId = void 0;
var _namespaces = require("../base/namespaces");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const toolIdRegexp = exports.toolIdRegexp = /^(?:[a-z0-9](?:[a-z0-9_-]*[a-z0-9])?)(?:\.(?:[a-z0-9](?:[a-z0-9_-]*[a-z0-9])?))*$/;
const toolIdMaxLength = exports.toolIdMaxLength = 64;
const reservedKeywords = ['new'];

/**
 * Check if the given ID is a reserved ID
 * Atm this only checks for `new` because that's a value we're using for url paths on the UI.
 */
const isReservedToolId = id => {
  return reservedKeywords.includes(id);
};

/**
 * Validate that a tool id has the right format,
 * returning an error message if it fails the validation,
 * and undefined otherwise.
 *
 * @param toolId: the toolId to validate
 * @param builtIn: set to true if we're validating a built-in (internal) tool id.
 */
exports.isReservedToolId = isReservedToolId;
const validateToolId = ({
  toolId,
  builtIn
}) => {
  if (!toolIdRegexp.test(toolId)) {
    return `Tool ids must start and end with a letter or number, and can only contain lowercase letters, numbers, dots, hyphens and underscores`;
  }
  if (toolId.length > toolIdMaxLength) {
    return `Tool ids are limited to ${toolIdMaxLength} characters.`;
  }
  if ((0, _namespaces.hasNamespaceName)(toolId)) {
    return `Tool id cannot have the same name as a reserved namespace.`;
  }
  if (!builtIn) {
    if ((0, _namespaces.isInProtectedNamespace)(toolId)) {
      return `Tool id is using a protected namespace.`;
    }
  }
};
exports.validateToolId = validateToolId;