"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSearchTerm = exports.getFieldValueMap = exports.applyAliases = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Return a name->values map for all the field clauses of given query.
 *
 * @example
 * ```
 *  getFieldValueMap(Query.parse('foo:bar foo:baz hello:dolly term'));
 *  >>  { foo: ['bar', 'baz'], hello: ['dolly] }
 * ```
 */
const getFieldValueMap = query => {
  const fieldMap = new Map();
  query.ast.clauses.forEach(clause => {
    if (clause.type === 'field') {
      var _fieldMap$get;
      const {
        field,
        value
      } = clause;
      fieldMap.set(field, [...((_fieldMap$get = fieldMap.get(field)) !== null && _fieldMap$get !== void 0 ? _fieldMap$get : []), ...(Array.isArray(value) ? value : [value])]);
    }
  });
  return fieldMap;
};

/**
 * Aggregate all term clauses from given query and concatenate them.
 */
exports.getFieldValueMap = getFieldValueMap;
const getSearchTerm = query => {
  var _term;
  let term;
  if (query.ast.getTermClauses().length) {
    term = query.ast.getTermClauses().map(clause => clause.value).join(' ').replace(/\s{2,}/g, ' ').trim();
  }
  return (_term = term) !== null && _term !== void 0 && _term.length ? term : undefined;
};

/**
 * Apply given alias map to the value map, concatenating the aliases values to the alias target, and removing
 * the alias entry. Any non-aliased entries will remain unchanged.
 *
 * @example
 * ```
 *  applyAliases({ field: ['foo'], alias: ['bar'], hello: ['dolly'] }, { field: ['alias']});
 *  >>  { field: ['foo', 'bar'], hello: ['dolly'] }
 * ```
 */
exports.getSearchTerm = getSearchTerm;
const applyAliases = (valueMap, aliasesMap) => {
  const reverseLookup = {};
  Object.entries(aliasesMap).forEach(([canonical, aliases]) => {
    aliases.forEach(alias => {
      reverseLookup[alias] = canonical;
    });
  });
  const resultMap = new Map();
  valueMap.forEach((values, field) => {
    var _reverseLookup$field, _resultMap$get;
    const targetKey = (_reverseLookup$field = reverseLookup[field]) !== null && _reverseLookup$field !== void 0 ? _reverseLookup$field : field;
    resultMap.set(targetKey, [...((_resultMap$get = resultMap.get(targetKey)) !== null && _resultMap$get !== void 0 ? _resultMap$get : []), ...values]);
  });
  return resultMap;
};
exports.applyAliases = applyAliases;