"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createAgentPolicyWithPackages = createAgentPolicyWithPackages;
var _agent_policies_helpers = require("../../common/services/agent_policies_helpers");
var _common = require("../../common");
var _ = require(".");
var _package_policies = require("./package_policies");
var _packages = require("./epm/packages");
var _api_keys = require("./api_keys");
var _agentless_agent = require("./agents/agentless_agent");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getFleetServerAgentPolicyId(soClient, agentPolicyService) {
  const logger = _.appContextService.getLogger().get('getFleetServerAgentPolicyId');
  logger.debug(`Retrieving fleet server agent policy id using soClient scoped to [${soClient.getCurrentNamespace()}]`);
  let agentPolicyId;
  // creating first fleet server policy with id '(space-)?fleet-server-policy'
  let agentPolicy;
  try {
    agentPolicy = await agentPolicyService.get(soClient, (0, _agent_policies_helpers.getDefaultFleetServerpolicyId)(soClient.getCurrentNamespace()), false);
  } catch (err) {
    if (!err.isBoom || err.output.statusCode !== 404) {
      throw err;
    }
  }
  if (!agentPolicy) {
    agentPolicyId = (0, _agent_policies_helpers.getDefaultFleetServerpolicyId)(soClient.getCurrentNamespace());
  }
  logger.debug(`Returning agent policy id [${agentPolicyId}]`);
  return agentPolicyId;
}
async function createPackagePolicy(soClient, esClient, agentPolicyService, agentPolicy, packageToInstall, options) {
  var _agentPolicy$space_id;
  const newPackagePolicy = await _.packagePolicyService.buildPackagePolicyFromPackage(soClient, packageToInstall).catch(async error => {
    // rollback agent policy on error
    await agentPolicyService.delete(soClient, esClient, agentPolicy.id, {
      force: true,
      user: options.user
    });
    throw error;
  });
  if (!newPackagePolicy) return;
  newPackagePolicy.policy_id = agentPolicy.id;
  newPackagePolicy.policy_ids = [agentPolicy.id];
  newPackagePolicy.name = await (0, _package_policies.incrementPackageName)(soClient, packageToInstall, (_agentPolicy$space_id = agentPolicy.space_ids) !== null && _agentPolicy$space_id !== void 0 ? _agentPolicy$space_id : [options.spaceId]);
  if (agentPolicy.supports_agentless) {
    newPackagePolicy.supports_agentless = agentPolicy.supports_agentless;
  }
  await _.packagePolicyService.create(soClient, esClient, newPackagePolicy, {
    spaceId: options.spaceId,
    user: options.user,
    bumpRevision: false,
    authorizationHeader: options.authorizationHeader,
    force: options.force
  });
}
async function createAgentPolicyWithPackages({
  soClient,
  esClient,
  agentPolicyService,
  newPolicy,
  hasFleetServer,
  withSysMonitoring,
  monitoringEnabled,
  spaceId,
  user,
  authorizationHeader,
  force,
  forcePackagePolicyCreation
}) {
  const logger = _.appContextService.getLogger().get('createAgentPolicyWithPackages');
  logger.debug(`creating policy [${newPolicy.name}] for space [${spaceId}] using soClient scoped to [${soClient.getCurrentNamespace()}]`);
  let agentPolicyId = newPolicy.id;
  const packagesToInstall = [];
  if (hasFleetServer) {
    packagesToInstall.push(_common.FLEET_SERVER_PACKAGE);
    agentPolicyId = agentPolicyId || (await getFleetServerAgentPolicyId(soClient, agentPolicyService));
    if (agentPolicyId === (0, _agent_policies_helpers.getDefaultFleetServerpolicyId)(spaceId)) {
      // setting first fleet server policy to default, so that fleet server can enroll without setting policy_id
      newPolicy.is_default_fleet_server = true;
    }
  }
  if (withSysMonitoring) {
    packagesToInstall.push(_common.FLEET_SYSTEM_PACKAGE);
  }
  if (monitoringEnabled !== null && monitoringEnabled !== void 0 && monitoringEnabled.length) {
    packagesToInstall.push(_common.FLEET_ELASTIC_AGENT_PACKAGE);
  }
  if (packagesToInstall.length > 0) {
    logger.debug(() => `Installing packages [${packagesToInstall.join(', ')}]`);
    await (0, _packages.bulkInstallPackages)({
      savedObjectsClient: soClient,
      esClient,
      packagesToInstall,
      spaceId,
      authorizationHeader,
      force
    });
  }
  const {
    id,
    ...policy
  } = newPolicy; // omit id from create object

  const agentPolicy = await agentPolicyService.create(soClient, esClient, policy, {
    user,
    id: agentPolicyId,
    authorizationHeader,
    hasFleetServer,
    skipDeploy: true // skip deploying the policy until package policies are added
  });

  // Since agentPolicyService does not handle multispace assignments, we need to keep this context with package policy creation
  const agentPolicyWithStagedSpaces = {
    ...agentPolicy,
    space_ids: newPolicy.space_ids
  };

  // Create the fleet server package policy and add it to agent policy.
  if (hasFleetServer) {
    await createPackagePolicy(soClient, esClient, agentPolicyService, agentPolicyWithStagedSpaces, _common.FLEET_SERVER_PACKAGE, {
      spaceId,
      user,
      authorizationHeader,
      force: force || forcePackagePolicyCreation
    });
  }

  // Create the system monitoring package policy and add it to agent policy.
  if (withSysMonitoring) {
    await createPackagePolicy(soClient, esClient, agentPolicyService, agentPolicyWithStagedSpaces, _common.FLEET_SYSTEM_PACKAGE, {
      spaceId,
      user,
      authorizationHeader,
      force: force || forcePackagePolicyCreation
    });
  }
  await (0, _api_keys.ensureDefaultEnrollmentAPIKeyForAgentPolicy)(soClient, esClient, agentPolicy.id);
  await agentPolicyService.deployPolicy(soClient, agentPolicy.id);

  // Create the agentless agent
  if (agentPolicy.supports_agentless) {
    try {
      await _agentless_agent.agentlessAgentService.createAgentlessAgent(esClient, soClient, agentPolicy);
    } catch (err) {
      await agentPolicyService.delete(soClient, esClient, agentPolicy.id).catch(() => {
        _.appContextService.getLogger().error(`Error deleting agentless policy`, {
          error: agentPolicy
        });
      });
      throw err;
    }
  }
  return agentPolicy;
}