"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getExceptionsPreCreateItemHandler = void 0;
var _artifacts = require("../../../../common/endpoint/service/artifacts");
var _errors = require("../validators/errors");
var _validators = require("../validators");
var _utils = require("../../../../common/endpoint/service/artifacts/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getExceptionsPreCreateItemHandler = endpointAppContext => {
  return async function ({
    data,
    context: {
      request
    }
  }) {
    if (data.namespaceType !== 'agnostic') {
      return data;
    }
    let isEndpointArtifact = false;
    let validatedItem = data;

    // Validate trusted apps
    if (_validators.TrustedAppValidator.isTrustedApp(data)) {
      isEndpointArtifact = true;
      const trustedAppValidator = new _validators.TrustedAppValidator(endpointAppContext, request);
      validatedItem = await trustedAppValidator.validatePreCreateItem(data);
      trustedAppValidator.notifyFeatureUsage(data, 'TRUSTED_APP_BY_POLICY');
    }

    // Validate trusted devices
    if (_validators.TrustedDeviceValidator.isTrustedDevice(data)) {
      isEndpointArtifact = true;
      const trustedDeviceValidator = new _validators.TrustedDeviceValidator(endpointAppContext, request);
      validatedItem = await trustedDeviceValidator.validatePreCreateItem(data);
      trustedDeviceValidator.notifyFeatureUsage(data, 'TRUSTED_DEVICE_BY_POLICY');
    }

    // Validate event filter
    if (_validators.EventFilterValidator.isEventFilter(data)) {
      isEndpointArtifact = true;
      const eventFilterValidator = new _validators.EventFilterValidator(endpointAppContext, request);
      validatedItem = await eventFilterValidator.validatePreCreateItem(data);
      eventFilterValidator.notifyFeatureUsage(data, 'EVENT_FILTERS_BY_POLICY');
    }

    // Validate host isolation
    if (_validators.HostIsolationExceptionsValidator.isHostIsolationException(data)) {
      isEndpointArtifact = true;
      const hostIsolationExceptionsValidator = new _validators.HostIsolationExceptionsValidator(endpointAppContext, request);
      validatedItem = await hostIsolationExceptionsValidator.validatePreCreateItem(data);
      hostIsolationExceptionsValidator.notifyFeatureUsage(data, 'HOST_ISOLATION_EXCEPTION_BY_POLICY');
      hostIsolationExceptionsValidator.notifyFeatureUsage(data, 'HOST_ISOLATION_EXCEPTION');
    }

    // Validate blocklists
    if (_validators.BlocklistValidator.isBlocklist(data)) {
      isEndpointArtifact = true;
      const blocklistValidator = new _validators.BlocklistValidator(endpointAppContext, request);
      validatedItem = await blocklistValidator.validatePreCreateItem(data);
      blocklistValidator.notifyFeatureUsage(data, 'BLOCKLIST_BY_POLICY');
    }

    // validate endpoint exceptions
    if (_validators.EndpointExceptionsValidator.isEndpointException(data)) {
      isEndpointArtifact = true;
      const endpointExceptionValidator = new _validators.EndpointExceptionsValidator(endpointAppContext, request);
      validatedItem = await endpointExceptionValidator.validatePreCreateItem(data);

      // If artifact does not have an assignment tag, then add it now. This is in preparation for
      // adding per-policy support to Endpoint Exceptions as well as to support space awareness
      if (!(0, _utils.hasGlobalOrPerPolicyTag)(validatedItem)) {
        var _validatedItem$tags;
        validatedItem.tags = (_validatedItem$tags = validatedItem.tags) !== null && _validatedItem$tags !== void 0 ? _validatedItem$tags : [];
        validatedItem.tags.push(_artifacts.GLOBAL_ARTIFACT_TAG);
      }
      endpointExceptionValidator.notifyFeatureUsage(data, 'ENDPOINT_EXCEPTIONS');
    }
    if (isEndpointArtifact && endpointAppContext.experimentalFeatures.endpointManagementSpaceAwarenessEnabled) {
      if (!request) {
        throw new _errors.EndpointArtifactExceptionValidationError(`Missing HTTP Request object`);
      }
      const spaceId = (await endpointAppContext.getActiveSpace(request)).id;
      (0, _utils.setArtifactOwnerSpaceId)(validatedItem, spaceId);
      return validatedItem;
    }
    return isEndpointArtifact ? validatedItem : data;
  };
};
exports.getExceptionsPreCreateItemHandler = getExceptionsPreCreateItemHandler;