"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerSiemDashboardMigrationsResourceUpsertRoute = void 0;
var _zodHelpers = require("@kbn/zod-helpers");
var _lodash = require("lodash");
var _constants = require("../../../../../../common/siem_migrations/dashboards/constants");
var _dashboard_migration = require("../../../../../../common/siem_migrations/model/api/dashboards/dashboard_migration.gen");
var _resources = require("../../../../../../common/siem_migrations/dashboards/resources");
var _audit = require("../../../common/api/util/audit");
var _authz = require("../../../common/api/util/authz");
var _with_license = require("../../../common/api/util/with_license");
var _lookups = require("../../../rules/api/util/lookups");
var _with_existing_migration_id = require("../../../common/api/util/with_existing_migration_id");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerSiemDashboardMigrationsResourceUpsertRoute = (router, logger) => {
  router.versioned.post({
    path: _constants.SIEM_DASHBOARD_MIGRATION_RESOURCES_PATH,
    access: 'internal',
    security: {
      authz: _authz.authz
    },
    options: {
      body: {
        maxBytes: 26214400
      }
    } // rise payload limit to 25MB
  }).addVersion({
    version: '1',
    validate: {
      request: {
        params: (0, _zodHelpers.buildRouteValidationWithZod)(_dashboard_migration.UpsertDashboardMigrationResourcesRequestParams),
        body: (0, _zodHelpers.buildRouteValidationWithZod)(_dashboard_migration.UpsertDashboardMigrationResourcesRequestBody)
      }
    }
  }, (0, _with_license.withLicense)((0, _with_existing_migration_id.withExistingMigration)(async (context, req, res) => {
    const resources = req.body;
    const migrationId = req.params.migration_id;
    const siemMigrationAuditLogger = new _audit.SiemMigrationAuditLogger(context.securitySolution, 'dashboards');
    try {
      const ctx = await context.resolve(['securitySolution']);
      const dashboardMigrationsClient = ctx.securitySolution.siemMigrations.getDashboardsClient();
      await siemMigrationAuditLogger.logUploadResources({
        migrationId
      });
      const [lookups, macros] = (0, _lodash.partition)(resources, {
        type: 'lookup'
      });
      const processedLookups = await (0, _lookups.processLookups)(lookups, dashboardMigrationsClient.data.lookups);
      // Add migration_id to all resources
      const resourcesUpsert = [...macros, ...processedLookups].map(resource => ({
        ...resource,
        migration_id: migrationId
      }));

      // Create identified resource documents to keep track of them (without content)
      const resourceIdentifier = new _resources.DashboardResourceIdentifier('splunk');
      const resourcesToCreate = resourceIdentifier.fromResources(resources).map(resource => ({
        ...resource,
        migration_id: migrationId
      }));
      await Promise.all([dashboardMigrationsClient.data.resources.upsert(resourcesUpsert), dashboardMigrationsClient.data.resources.create(resourcesToCreate)]);
      return res.ok({
        body: {
          acknowledged: true
        }
      });
    } catch (error) {
      logger.error(error);
      await siemMigrationAuditLogger.logUploadResources({
        migrationId,
        error
      });
      return res.customError({
        statusCode: 500,
        body: error.message
      });
    }
  })));
};
exports.registerSiemDashboardMigrationsResourceUpsertRoute = registerSiemDashboardMigrationsResourceUpsertRoute;