"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PadPackageInstallationClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _packages = require("@kbn/fleet-plugin/server/services/epm/packages");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class PadPackageInstallationClient {
  constructor(opts) {
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "soClient", void 0);
    /**
     * A type guard function to determine if the bulk package installation response is an error type
     *
     * @param potentialInstallResponseError the `BulkInstallResponse` to check type of
     */
    (0, _defineProperty2.default)(this, "isInstallError", potentialInstallResponseError => {
      return potentialInstallResponseError.error !== undefined;
    });
    this.opts = opts;
    this.esClient = opts.clusterClient.asCurrentUser;
    this.soClient = opts.soClient;
  }
  log(level, msg) {
    this.opts.logger[level](`[Privileged access detection] [namespace: ${this.opts.namespace}] ${msg}`);
  }
  async getCurrentlyInstalledPADPackage() {
    const installedPadPackages = await (0, _packages.getInstalledPackages)({
      savedObjectsClient: this.soClient,
      esClient: this.esClient,
      nameQuery: 'pad',
      perPage: 100,
      sortOrder: 'asc'
    });
    return installedPadPackages.items.find(installedPackage => installedPackage.name === 'pad');
  }
  async getJobs() {
    const jobs = (await this.esClient.ml.getJobs({
      job_id: 'pad'
    })).jobs.filter(each => each.custom_settings.created_by === 'ml-module-pad');
    const jobStatsByJobId = (await this.esClient.ml.getJobStats({
      job_id: 'pad'
    })).jobs.reduce((accumulator, nextJobStats) => ({
      ...accumulator,
      [nextJobStats.job_id]: nextJobStats
    }), {});
    return jobs.map(eachJob => ({
      job_id: eachJob.job_id,
      description: eachJob.description,
      state: jobStatsByJobId[eachJob.job_id].state
    }));
  }
  async getStatus() {
    const packageInstalled = !!(await this.getCurrentlyInstalledPADPackage());
    const packageInstallationStatus = packageInstalled ? 'complete' : 'incomplete';
    if (!packageInstalled) {
      // even if there happen to be jobs that match our search criteria, if the package is not installed, we consider the ML installation incomplete and the jobs to not be associated with our privileged access detection usage
      return {
        package_installation_status: packageInstallationStatus,
        ml_module_setup_status: 'incomplete',
        jobs: []
      };
    }
    try {
      const jobs = await this.getJobs();
      const mlModuleSetupStatus = jobs.length > 0 ? 'complete' : 'incomplete';
      return {
        package_installation_status: packageInstallationStatus,
        ml_module_setup_status: mlModuleSetupStatus,
        jobs
      };
    } catch (e) {
      this.log('info', 'The privileged access detection package is installed, but the ML jobs are not yet set up.');
      return {
        package_installation_status: packageInstallationStatus,
        ml_module_setup_status: 'incomplete',
        jobs: []
      };
    }
  }
  async getPrivilegedAccessDetectionPackageFromRegistry() {
    return (await (0, _packages.getPackages)({
      savedObjectsClient: this.soClient,
      category: 'security',
      prerelease: true
    })).find(availablePackage => availablePackage.name === 'pad');
  }
  async installPrivilegedAccessDetectionPackage() {
    const alreadyInstalledPadPackage = await this.getCurrentlyInstalledPADPackage();
    if (alreadyInstalledPadPackage) {
      return {
        message: 'Privileged access detection package was already installed.'
      };
    }
    const availablePadPackage = await this.getPrivilegedAccessDetectionPackageFromRegistry();
    if (!availablePadPackage) {
      this.log('info', 'Privileged access detection package was not found');
      throw new Error('Privileged access detection package was not found.');
    }
    const installationResponse = await this.installPackage(availablePadPackage);
    if (!installationResponse || this.isInstallError(installationResponse)) {
      throw new Error(`Failed to install privileged access detection package. ${installationResponse === null || installationResponse === void 0 ? void 0 : installationResponse.error}`);
    }
    return {
      message: 'Successfully installed privileged access detection package.'
    };
  }
  async installPackage(installablePackage) {
    this.log('info', `Installing Privileged Access Detection package: ${installablePackage.name} ${installablePackage.version}`);
    const bulkInstallResponse = await (0, _packages.bulkInstallPackages)({
      savedObjectsClient: this.soClient,
      packagesToInstall: [{
        name: installablePackage.name,
        version: installablePackage.version,
        prerelease: true
      }],
      esClient: this.esClient,
      spaceId: this.opts.namespace,
      skipIfInstalled: true,
      force: true
    });
    return bulkInstallResponse.length > 0 ? bulkInstallResponse[0] : undefined;
  }
}
exports.PadPackageInstallationClient = PadPackageInstallationClient;