"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SentinelOneActionsClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _constants = require("@kbn/stack-connectors-plugin/common/sentinelone/constants");
var _lodash = require("lodash");
var _index_name_utilities = require("../../../../../../common/endpoint/utils/index_name_utilities");
var _sentinel_one = require("../../../../../../common/endpoint/service/response_actions/sentinel_one");
var _constants2 = require("../../constants");
var _common = require("../../../../../../common");
var _utils = require("../../../../utils");
var _constants3 = require("../../../../../../common/endpoint/service/response_actions/constants");
var _stringify = require("../../../../utils/stringify");
var _errors = require("../errors");
var _base_response_actions_client = require("../lib/base_response_actions_client");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* eslint-disable @typescript-eslint/no-explicit-any */

const NOOP_THROW = () => {
  throw new _errors.ResponseActionsClientError('not implemented!');
};
class SentinelOneActionsClient extends _base_response_actions_client.ResponseActionsClientImpl {
  constructor({
    connectorActions,
    ...options
  }) {
    super(options);
    (0, _defineProperty2.default)(this, "agentType", 'sentinel_one');
    (0, _defineProperty2.default)(this, "connectorActionsClient", void 0);
    this.connectorActionsClient = connectorActions;
    connectorActions.setup(_constants.SENTINELONE_CONNECTOR_ID);
  }
  async fetchSentinelOneAgentIndexNames() {
    const cachedInfo = this.cache.get('fetchSentinelOneAgentIndexNames');
    if (cachedInfo) {
      this.log.debug(`Returning cached response with list of Agent index names:\n${(0, _stringify.stringify)(cachedInfo)}`);
      return cachedInfo;
    }
    const fleetServices = this.options.endpointService.getInternalFleetServices(this.options.spaceId);
    const indexNamespaces = await fleetServices.getIntegrationNamespaces(['sentinel_one']);
    const indexNames = [];
    for (const namespaces of Object.values(indexNamespaces)) {
      if (namespaces.length > 0) {
        indexNames.push(...namespaces.map(namespace => (0, _index_name_utilities.buildIndexNameWithNamespace)(_sentinel_one.SENTINEL_ONE_AGENT_INDEX_PATTERN, namespace)));
      }
    }
    this.log.debug(() => `Index list with namespace:\n${(0, _stringify.stringify)(indexNames)}`);
    this.cache.set('fetchSentinelOneAgentIndexNames', indexNames);
    return indexNames;
  }
  async fetchAgentPolicyInfo(agentIds) {
    const cacheKey = `fetchAgentPolicyInfo:${agentIds.sort().join('#')}`;
    const cacheResponse = this.cache.get(cacheKey);
    if (cacheResponse) {
      this.log.debug(() => `Cached agent policy info. found - returning it:\n${(0, _stringify.stringify)(cacheResponse)}`);
      return cacheResponse;
    }
    const esClient = this.options.esClient;
    const esSearchRequest = {
      index: await this.fetchSentinelOneAgentIndexNames(),
      query: {
        bool: {
          filter: [{
            terms: {
              'sentinel_one.agent.agent.id': agentIds
            }
          }]
        }
      },
      collapse: {
        field: 'sentinel_one.agent.agent.id',
        inner_hits: {
          name: 'most_recent',
          size: 1,
          _source: ['agent', 'sentinel_one.agent.agent.id', 'event.created'],
          sort: [{
            'event.created': 'desc'
          }]
        }
      },
      _source: false,
      ignore_unavailable: true
    };
    if (!esSearchRequest.index || esSearchRequest.index.length === 0) {
      throw new _errors.ResponseActionsClientError(`Unable to build list of indexes while retrieving policy information for SentinelOne agents [${agentIds.join(', ')}]. Check to ensure at least one integration policy exists.`, 400);
    }
    this.log.debug(() => `Looking for agents with:\n${(0, _stringify.stringify)(esSearchRequest)}`);

    // Get the latest ingested document for each agent ID
    const s1AgentsEsResults = await esClient.search(esSearchRequest).catch(_utils.catchAndWrapError);
    this.log.debug(() => `SentinelOne Agent records found:\n${(0, _stringify.stringify)(s1AgentsEsResults, 20)}`);
    const agentIdsFound = [];
    const fleetAgentIdToS1AgentIdMap = s1AgentsEsResults.hits.hits.reduce((acc, s1AgentEsDoc) => {
      var _s1AgentEsDoc$inner_h;
      const doc = (_s1AgentEsDoc$inner_h = s1AgentEsDoc.inner_hits) === null || _s1AgentEsDoc$inner_h === void 0 ? void 0 : _s1AgentEsDoc$inner_h.most_recent.hits.hits[0]._source;
      if (doc) {
        agentIdsFound.push(doc.sentinel_one.agent.agent.id);
        acc[doc.agent.id] = doc.sentinel_one.agent.agent.id;
      }
      return acc;
    }, {});
    const elasticAgentIds = Object.keys(fleetAgentIdToS1AgentIdMap);
    if (elasticAgentIds.length === 0) {
      throw new _errors.ResponseActionsClientError(`Unable to find elastic agent IDs for SentinelOne agent ids: [${agentIds.join(', ')}]`, 400);
    }

    // ensure all agent ids were found
    for (const agentId of agentIds) {
      if (!agentIdsFound.includes(agentId)) {
        throw new _errors.ResponseActionsClientError(`SentinelOne agent id [${agentId}] not found`, 404);
      }
    }
    const agentPolicyInfo = await this.fetchFleetInfoForAgents(elasticAgentIds);
    for (const agentInfo of agentPolicyInfo) {
      agentInfo.agentId = fleetAgentIdToS1AgentIdMap[agentInfo.elasticAgentId];
    }
    this.cache.set(cacheKey, agentPolicyInfo);
    return agentPolicyInfo;
  }
  async handleResponseActionCreation(reqIndexOptions) {
    const actionRequestDoc = await this.writeActionRequestToEndpointIndex(reqIndexOptions);
    await this.updateCases({
      command: reqIndexOptions.command,
      caseIds: reqIndexOptions.case_ids,
      alertIds: reqIndexOptions.alert_ids,
      actionId: actionRequestDoc.EndpointActions.action_id,
      hosts: reqIndexOptions.endpoint_ids.map(agentId => {
        var _actionRequestDoc$End, _actionRequestDoc$End2;
        return {
          hostId: agentId,
          hostname: (_actionRequestDoc$End = (_actionRequestDoc$End2 = actionRequestDoc.EndpointActions.data.hosts) === null || _actionRequestDoc$End2 === void 0 ? void 0 : _actionRequestDoc$End2[agentId].name) !== null && _actionRequestDoc$End !== void 0 ? _actionRequestDoc$End : ''
        };
      }),
      comment: reqIndexOptions.comment
    });
    return {
      actionEsDoc: actionRequestDoc,
      actionDetails: await this.fetchActionDetails(actionRequestDoc.EndpointActions.action_id)
    };
  }
  async writeActionRequestToEndpointIndex(actionRequest) {
    var _actionRequest$meta;
    const agentId = actionRequest.endpoint_ids[0];
    const agentDetails = await this.getAgentDetails(agentId);
    const doc = await super.writeActionRequestToEndpointIndex({
      ...actionRequest,
      hosts: {
        [agentId]: {
          name: agentDetails.computerName
        }
      },
      meta: {
        // Add common meta data
        agentUUID: agentId,
        agentId: agentDetails.id,
        hostName: agentDetails.computerName,
        ...((_actionRequest$meta = actionRequest.meta) !== null && _actionRequest$meta !== void 0 ? _actionRequest$meta : {})
      }
    });
    return doc;
  }

  /**
   * Sends actions to SentinelOne directly (via Connector)
   * @internal
   */
  async sendAction(actionType, actionParams) {
    const executeOptions = {
      params: {
        subAction: actionType,
        subActionParams: actionParams
      }
    };
    this.log.debug(() => `calling connector actions 'execute()' for SentinelOne with:\n${(0, _stringify.stringify)(executeOptions)}`);
    const actionSendResponse = await this.connectorActionsClient.execute(executeOptions);
    if (actionSendResponse.status === 'error') {
      this.log.error((0, _stringify.stringify)(actionSendResponse));
      throw new _errors.ResponseActionsClientError(`Attempt to send [${actionType}] to SentinelOne failed: ${actionSendResponse.serviceMessage || actionSendResponse.message}`, 500, actionSendResponse);
    }
    this.log.debug(() => `Response:\n${(0, _stringify.stringify)(actionSendResponse)}`);
    return actionSendResponse;
  }

  /** Gets agent details directly from SentinelOne */
  async getAgentDetails(agentId) {
    var _await$this$sendActio;
    const cacheKey = `getAgentDetails:${agentId}`;
    const cachedEntry = this.cache.get(cacheKey);
    if (cachedEntry) {
      this.log.debug(`Found cached agent details for UUID [${agentId}]:\n${(0, _stringify.stringify)(cachedEntry)}`);
      return cachedEntry;
    }
    let agentDetailsRetrievalError = '';
    const s1ApiResponse = (_await$this$sendActio = await this.sendAction(_constants.SUB_ACTION.GET_AGENTS, {
      ids: agentId
    }).catch(err => {
      agentDetailsRetrievalError = err.message;
    })) === null || _await$this$sendActio === void 0 ? void 0 : _await$this$sendActio.data;
    if (!s1ApiResponse || !s1ApiResponse.data[0]) {
      throw new _errors.ResponseActionsClientError(`SentinelOne agent id [${agentId}] not found${agentDetailsRetrievalError ? `. API response: ${agentDetailsRetrievalError}` : ''}`, 404);
    }
    this.cache.set(cacheKey, s1ApiResponse.data[0]);
    return s1ApiResponse.data[0];
  }
  async validateRequest(payload) {
    // TODO:PT support multiple agents
    if (payload.endpoint_ids.length > 1) {
      return {
        isValid: false,
        error: new _errors.ResponseActionsClientError(`[body.endpoint_ids]: Multiple agents IDs not currently supported for SentinelOne`, 400)
      };
    }

    // Ensure Agent ID is valid
    const agentId = payload.endpoint_ids[0];
    const agentDetails = await this.getAgentDetails(agentId);
    if (agentDetails.id !== payload.endpoint_ids[0]) {
      throw new _errors.ResponseActionsClientError(`Agent id [${agentId}] not found`, 404);
    }

    // KILL-PROCESS:
    // validate that we have a `process_name`. We need this here because the schema for this command
    // specifically because `KillProcessRequestBody` allows 3 types of parameters.
    if (payload.command === 'kill-process') {
      const parameters = payload.parameters;
      if (!parameters || !('process_name' in parameters) || !parameters.process_name) {
        return {
          isValid: false,
          error: new _errors.ResponseActionsClientError('[body.parameters.process_name]: missing parameter or value is empty')
        };
      }
    }

    // RUN SCRIPT
    if (payload.command === 'runscript') {
      var _parameters$scriptId, _runScriptRequestPayl;
      const runScriptRequestPayload = payload;
      const parameters = runScriptRequestPayload.parameters;
      const scriptId = ((_parameters$scriptId = parameters === null || parameters === void 0 ? void 0 : parameters.scriptId) !== null && _parameters$scriptId !== void 0 ? _parameters$scriptId : '').trim();
      if (!scriptId) {
        throw new _errors.ResponseActionsClientError(`[parameters.scriptId]: missing parameter or value is empty`);
      }
      const scriptDetails = await this.sendAction(_constants.SUB_ACTION.GET_REMOTE_SCRIPTS, {
        ids: scriptId
      }).then(response => {
        var _response$data;
        return (_response$data = response.data) === null || _response$data === void 0 ? void 0 : _response$data.data[0];
      });
      if (!scriptDetails || scriptDetails.id !== scriptId) {
        throw new _errors.ResponseActionsClientError(`Script ID [${scriptId}] not found`, 404);
      }

      // This validation is needed because, when calling the S1 API to execute a script ID that does not support
      // the targeted host's OS type, S1 API still returns a successful response, but it includes
      // no `parentTaskId` - which indicates nothing was actually triggered in S1.
      if (agentDetails.osType && !scriptDetails.osTypes.includes(agentDetails.osType)) {
        throw new _errors.ResponseActionsClientError(`Script [${scriptDetails.scriptName}] not supported for OS type [${agentDetails.osType}]`, 400);
      }

      // Prepend the script name to the `comment` field for reference
      const scriptNameComment = `(Script name: ${scriptDetails.scriptName})`;
      if (!((_runScriptRequestPayl = runScriptRequestPayload.comment) !== null && _runScriptRequestPayl !== void 0 ? _runScriptRequestPayl : '').startsWith(scriptNameComment)) {
        runScriptRequestPayload.comment = `(Script name: ${scriptDetails.scriptName})${runScriptRequestPayload.comment ? ` ${runScriptRequestPayload.comment}` : ''}`;
      }
    }
    return super.validateRequest(payload);
  }
  async isolate(actionRequest, options = {}) {
    const reqIndexOptions = {
      ...actionRequest,
      ...this.getMethodOptions(options),
      parameters: undefined,
      command: 'isolate'
    };
    if (!reqIndexOptions.error) {
      var _error;
      let error = (await this.validateRequest(reqIndexOptions)).error;
      if (!error) {
        try {
          await this.sendAction(_constants.SUB_ACTION.ISOLATE_HOST, {
            ids: actionRequest.endpoint_ids[0]
          });
        } catch (err) {
          error = err;
        }
      }
      reqIndexOptions.error = (_error = error) === null || _error === void 0 ? void 0 : _error.message;
      if (!this.options.isAutomated && error) {
        throw error;
      }
    }
    const {
      actionDetails,
      actionEsDoc: actionRequestDoc
    } = await this.handleResponseActionCreation(reqIndexOptions);
    if (!actionRequestDoc.error && !this.options.endpointService.experimentalFeatures.responseActionsSentinelOneV2Enabled) {
      await this.writeActionResponseToEndpointIndex({
        actionId: actionRequestDoc.EndpointActions.action_id,
        agentId: actionRequestDoc.agent.id,
        data: {
          command: actionRequestDoc.EndpointActions.data.command
        }
      });
      return this.fetchActionDetails(actionRequestDoc.EndpointActions.action_id);
    }
    return actionDetails;
  }
  async release(actionRequest, options = {}) {
    const reqIndexOptions = {
      ...actionRequest,
      ...this.getMethodOptions(options),
      parameters: undefined,
      command: 'unisolate'
    };
    if (!reqIndexOptions.error) {
      var _error2;
      let error = (await this.validateRequest(reqIndexOptions)).error;
      if (!error) {
        try {
          await this.sendAction(_constants.SUB_ACTION.RELEASE_HOST, {
            ids: actionRequest.endpoint_ids[0]
          });
        } catch (err) {
          error = err;
        }
      }
      reqIndexOptions.error = (_error2 = error) === null || _error2 === void 0 ? void 0 : _error2.message;
      if (!this.options.isAutomated && error) {
        throw error;
      }
    }
    const {
      actionDetails,
      actionEsDoc: actionRequestDoc
    } = await this.handleResponseActionCreation(reqIndexOptions);
    if (!actionRequestDoc.error && !this.options.endpointService.experimentalFeatures.responseActionsSentinelOneV2Enabled) {
      await this.writeActionResponseToEndpointIndex({
        actionId: actionRequestDoc.EndpointActions.action_id,
        agentId: actionRequestDoc.agent.id,
        data: {
          command: actionRequestDoc.EndpointActions.data.command
        }
      });
      return this.fetchActionDetails(actionRequestDoc.EndpointActions.action_id);
    }
    return actionDetails;
  }
  async getFile(actionRequest, options) {
    if (!this.options.endpointService.experimentalFeatures.responseActionsSentinelOneGetFileEnabled) {
      throw new _errors.ResponseActionsClientError(`get-file not supported for ${this.agentType} agent type. Feature disabled`, 400);
    }
    const reqIndexOptions = {
      ...actionRequest,
      ...this.getMethodOptions(options),
      command: 'get-file'
    };
    const agentId = actionRequest.endpoint_ids[0];
    if (!reqIndexOptions.error) {
      var _error3;
      let error = (await this.validateRequest(reqIndexOptions)).error;
      const timestamp = new Date().toISOString();
      if (!error) {
        try {
          await this.sendAction(_constants.SUB_ACTION.FETCH_AGENT_FILES, {
            agentId: actionRequest.endpoint_ids[0],
            files: [actionRequest.parameters.path],
            zipPassCode: _constants3.RESPONSE_ACTIONS_ZIP_PASSCODE.sentinel_one
          });
        } catch (err) {
          error = err;
        }
      }
      reqIndexOptions.error = (_error3 = error) === null || _error3 === void 0 ? void 0 : _error3.message;
      if (!this.options.isAutomated && error) {
        throw error;
      }
      if (!error) {
        var _activityLogSearchRes;
        const activitySearchCriteria = {
          // Activity type for fetching a file from a host machine in SentinelOne:
          // {
          //   "id": 81
          //   "action": "User Requested Fetch Files",
          //   "descriptionTemplate": "The management user {{ username }} initiated a fetch file command to the agent {{ computer_name }} ({{ external_ip }}).",
          // },
          activityTypes: '81',
          limit: 1,
          sortBy: 'createdAt',
          sortOrder: 'asc',
          // eslint-disable-next-line @typescript-eslint/naming-convention
          createdAt__gte: timestamp,
          agentIds: agentId
        };

        // Fetch the Activity log entry for this get-file request and store needed data
        const activityLogSearchResponse = await this.sendAction(_constants.SUB_ACTION.GET_ACTIVITIES, activitySearchCriteria);
        if ((_activityLogSearchRes = activityLogSearchResponse.data) !== null && _activityLogSearchRes !== void 0 && _activityLogSearchRes.data.length) {
          var _activityLogSearchRes2;
          const activityLogItem = (_activityLogSearchRes2 = activityLogSearchResponse.data) === null || _activityLogSearchRes2 === void 0 ? void 0 : _activityLogSearchRes2.data[0];
          reqIndexOptions.meta = {
            commandBatchUuid: activityLogItem === null || activityLogItem === void 0 ? void 0 : activityLogItem.data.commandBatchUuid,
            activityId: activityLogItem === null || activityLogItem === void 0 ? void 0 : activityLogItem.id
          };
        } else {
          this.log.warn(`Unable to find a fetch file command entry in SentinelOne activity log. May be unable to complete response action. Search criteria used:\n${(0, _stringify.stringify)(activitySearchCriteria)}`);
        }
      }
    }
    return (await this.handleResponseActionCreation(reqIndexOptions)).actionDetails;
  }
  async getFileInfo(actionId, agentId) {
    await this.ensureValidActionId(actionId);
    const {
      EndpointActions: {
        data: {
          command
        }
      }
    } = await this.fetchActionRequestEsDoc(actionId);
    const {
      responseActionsSentinelOneGetFileEnabled: isGetFileEnabled,
      responseActionsSentinelOneProcessesEnabled: isRunningProcessesEnabled,
      responseActionsSentinelOneRunScriptEnabled: isRunScriptEnabled
    } = this.options.endpointService.experimentalFeatures;
    if (command === 'get-file' && !isGetFileEnabled || command === 'running-processes' && !isRunningProcessesEnabled || command === 'runscript' && !isRunScriptEnabled) {
      throw new _errors.ResponseActionsClientError(`File downloads are not supported for ${this.agentType} ${command}. Feature disabled`, 400);
    }
    const fileInfo = {
      actionId,
      agentId,
      id: agentId,
      agentType: this.agentType,
      status: 'AWAITING_UPLOAD',
      created: '',
      name: '',
      size: 0,
      mimeType: ''
    };
    try {
      switch (command) {
        case 'get-file':
          {
            var _agentResponse$meta$c, _agentResponse$meta, _agentResponse$meta$f, _agentResponse$meta2;
            const agentResponse = await this.fetchEsResponseDocForAgentId(actionId, agentId);

            // Unfortunately, there is no way to determine if a file is still available in SentinelOne without actually
            // calling the download API, which would return the following error:
            // {  "errors":[ {
            //      "code":4100010,
            //      "detail":"The requested files do not exist. Fetched files are deleted after 3 days, or earlier if more than 30 files are fetched.",
            //      "title":"Resource not found"
            // } ] }
            fileInfo.status = 'READY';
            fileInfo.created = (_agentResponse$meta$c = (_agentResponse$meta = agentResponse.meta) === null || _agentResponse$meta === void 0 ? void 0 : _agentResponse$meta.createdAt) !== null && _agentResponse$meta$c !== void 0 ? _agentResponse$meta$c : '';
            fileInfo.name = (_agentResponse$meta$f = (_agentResponse$meta2 = agentResponse.meta) === null || _agentResponse$meta2 === void 0 ? void 0 : _agentResponse$meta2.filename) !== null && _agentResponse$meta$f !== void 0 ? _agentResponse$meta$f : '';
            fileInfo.mimeType = 'application/octet-stream';
          }
          break;

        // Response Actions that use S1 scripts
        case 'running-processes':
        case 'runscript':
          {
            var _agentResponse$meta$t, _agentResponse$meta3, _s1ScriptResultsApiRe;
            const agentResponse = await this.fetchEsResponseDocForAgentId(actionId, agentId);
            const s1TaskId = (_agentResponse$meta$t = (_agentResponse$meta3 = agentResponse.meta) === null || _agentResponse$meta3 === void 0 ? void 0 : _agentResponse$meta3.taskId) !== null && _agentResponse$meta$t !== void 0 ? _agentResponse$meta$t : '';
            fileInfo.created = agentResponse['@timestamp'];
            const {
              data: s1ScriptResultsApiResponse
            } = await this.sendAction(_constants.SUB_ACTION.GET_REMOTE_SCRIPT_RESULTS, {
              taskIds: [s1TaskId]
            });
            const fileDownloadLink = ((_s1ScriptResultsApiRe = s1ScriptResultsApiResponse === null || s1ScriptResultsApiResponse === void 0 ? void 0 : s1ScriptResultsApiResponse.data.download_links) !== null && _s1ScriptResultsApiRe !== void 0 ? _s1ScriptResultsApiRe : []).find(linkInfo => {
              return linkInfo.taskId === s1TaskId;
            });
            if (!fileDownloadLink) {
              this.log.debug(`No download link found in SentinelOne for Task Id [${s1TaskId}]. Setting file status to DELETED. (Response action: ${actionId}, Agent Id: {${agentId})`);
              fileInfo.status = 'DELETED';
            } else {
              var _fileDownloadLink$fil;
              fileInfo.status = 'READY';
              fileInfo.name = (_fileDownloadLink$fil = fileDownloadLink.fileName) !== null && _fileDownloadLink$fil !== void 0 ? _fileDownloadLink$fil : `${actionId}-${agentId}.zip`;
              fileInfo.mimeType = 'application/octet-stream';
            }
          }
          break;
        default:
          throw new _errors.ResponseActionsClientError(`${command} does not support file downloads`, 400);
      }
    } catch (e) {
      // Ignore "no response doc" error for the agent and just return the file info with the status of 'AWAITING_UPLOAD'
      if (!(e instanceof _errors.ResponseActionAgentResponseEsDocNotFound)) {
        throw e;
      }
    }
    return fileInfo;
  }
  async getFileDownload(actionId, agentId) {
    await this.ensureValidActionId(actionId);
    const {
      EndpointActions: {
        data: {
          command
        }
      }
    } = await this.fetchActionRequestEsDoc(actionId);
    const {
      responseActionsSentinelOneGetFileEnabled: isGetFileEnabled,
      responseActionsSentinelOneProcessesEnabled: isRunningProcessesEnabled,
      responseActionsSentinelOneRunScriptEnabled: isRunScriptEnabled
    } = this.options.endpointService.experimentalFeatures;
    if (command === 'get-file' && !isGetFileEnabled || command === 'running-processes' && !isRunningProcessesEnabled || command === 'runscript' && !isRunScriptEnabled) {
      throw new _errors.ResponseActionsClientError(`File downloads are not supported for ${this.agentType} ${command}. Feature disabled`, 400);
    }
    let downloadStream;
    let fileName = 'download.zip';
    try {
      switch (command) {
        case 'get-file':
          {
            var _getFileAgentResponse, _getFileAgentResponse2;
            const getFileAgentResponse = await this.fetchEsResponseDocForAgentId(actionId, agentId);
            if (!((_getFileAgentResponse = getFileAgentResponse.meta) !== null && _getFileAgentResponse !== void 0 && _getFileAgentResponse.activityLogEntryId)) {
              throw new _errors.ResponseActionsClientError(`Unable to retrieve file from SentinelOne. Response ES document is missing [meta.activityLogEntryId]`);
            }
            const downloadAgentFileMethodOptions = {
              agentId,
              activityId: (_getFileAgentResponse2 = getFileAgentResponse.meta) === null || _getFileAgentResponse2 === void 0 ? void 0 : _getFileAgentResponse2.activityLogEntryId
            };
            const {
              data
            } = await this.sendAction(_constants.SUB_ACTION.DOWNLOAD_AGENT_FILE, downloadAgentFileMethodOptions);
            if (data) {
              downloadStream = data;
              fileName = getFileAgentResponse.meta.filename;
            }
          }
          break;
        case 'running-processes':
        case 'runscript':
          {
            var _processesAgentRespon, _processesAgentRespon2;
            const processesAgentResponse = await this.fetchEsResponseDocForAgentId(actionId, agentId);
            if (!((_processesAgentRespon = processesAgentResponse.meta) !== null && _processesAgentRespon !== void 0 && _processesAgentRespon.taskId)) {
              throw new _errors.ResponseActionsClientError(`Unable to retrieve file from SentinelOne for Response Action [${actionId}]. Response ES document is missing [meta.taskId]`);
            }
            const {
              data
            } = await this.sendAction(_constants.SUB_ACTION.DOWNLOAD_REMOTE_SCRIPT_RESULTS, {
              taskId: (_processesAgentRespon2 = processesAgentResponse.meta) === null || _processesAgentRespon2 === void 0 ? void 0 : _processesAgentRespon2.taskId
            });
            if (data) {
              downloadStream = data;
              fileName = `${actionId}_${agentId}.zip`;
            }
          }
          break;
        default:
          throw new _errors.ResponseActionsClientError(`${command} does not support file downloads`, 400);
      }
      if (!downloadStream) {
        throw new _errors.ResponseActionsClientError(`Unable to establish a file download Readable stream with SentinelOne for response action [${command}] [${actionId}]`);
      }
    } catch (e) {
      this.log.debug(() => `Attempt to get file download stream from SentinelOne for response action failed with:\n${(0, _stringify.stringify)(e)}`);
      throw e;
    }
    return {
      stream: downloadStream,
      mimeType: undefined,
      fileName
    };
  }
  async killProcess(actionRequest, options) {
    if (!this.options.endpointService.experimentalFeatures.responseActionsSentinelOneKillProcessEnabled) {
      throw new _errors.ResponseActionsClientError(`kill-process not supported for ${this.agentType} agent type. Feature disabled`, 400);
    }
    const reqIndexOptions = {
      ...actionRequest,
      ...this.getMethodOptions(options),
      command: 'kill-process',
      parameters: actionRequest.parameters,
      meta: {
        parentTaskId: ''
      }
    };
    if (!reqIndexOptions.error) {
      var _error4;
      let error = (await this.validateRequest(reqIndexOptions)).error;
      if (!error) {
        const s1AgentDetails = await this.getAgentDetails(reqIndexOptions.endpoint_ids[0]);
        const terminateScriptInfo = await this.fetchScriptInfo('kill-process', s1AgentDetails.osType);
        try {
          var _s1Response$data$data, _s1Response$data, _s1Response$data$data2;
          const s1Response = await this.sendAction(_constants.SUB_ACTION.EXECUTE_SCRIPT, {
            filter: {
              ids: actionRequest.endpoint_ids[0]
            },
            script: {
              scriptId: terminateScriptInfo.scriptId,
              taskDescription: this.buildExternalComment(reqIndexOptions),
              requiresApproval: false,
              outputDestination: 'SentinelCloud',
              inputParams: terminateScriptInfo.buildScriptArgs({
                processName: reqIndexOptions.parameters.process_name
              })
            }
          });
          reqIndexOptions.meta = {
            parentTaskId: (_s1Response$data$data = (_s1Response$data = s1Response.data) === null || _s1Response$data === void 0 ? void 0 : (_s1Response$data$data2 = _s1Response$data.data) === null || _s1Response$data$data2 === void 0 ? void 0 : _s1Response$data$data2.parentTaskId) !== null && _s1Response$data$data !== void 0 ? _s1Response$data$data : ''
          };
        } catch (err) {
          error = err;
        }
      }
      reqIndexOptions.error = (_error4 = error) === null || _error4 === void 0 ? void 0 : _error4.message;
      if (!this.options.isAutomated && error) {
        throw error;
      }
    }
    const {
      actionDetails
    } = await this.handleResponseActionCreation(reqIndexOptions);
    return actionDetails;
  }
  async runningProcesses(actionRequest, options) {
    if (!this.options.endpointService.experimentalFeatures.responseActionsSentinelOneProcessesEnabled) {
      throw new _errors.ResponseActionsClientError(`processes not supported for ${this.agentType} agent type. Feature disabled`, 400);
    }
    const reqIndexOptions = {
      ...actionRequest,
      ...this.getMethodOptions(options),
      parameters: undefined,
      command: 'running-processes',
      meta: {
        parentTaskId: ''
      }
    };
    if (!reqIndexOptions.error) {
      var _error5;
      let error = (await this.validateRequest(reqIndexOptions)).error;
      if (!error) {
        const s1AgentDetails = await this.getAgentDetails(reqIndexOptions.endpoint_ids[0]);
        const processesScriptInfo = await this.fetchScriptInfo('running-processes', s1AgentDetails.osType);
        try {
          var _s1Response$data$data3, _s1Response$data2, _s1Response$data2$dat;
          const s1Response = await this.sendAction(_constants.SUB_ACTION.EXECUTE_SCRIPT, {
            filter: {
              ids: actionRequest.endpoint_ids[0]
            },
            script: {
              scriptId: processesScriptInfo.scriptId,
              taskDescription: this.buildExternalComment(reqIndexOptions),
              requiresApproval: false,
              outputDestination: 'SentinelCloud',
              inputParams: processesScriptInfo.buildScriptArgs({})
            }
          });
          reqIndexOptions.meta = {
            parentTaskId: (_s1Response$data$data3 = (_s1Response$data2 = s1Response.data) === null || _s1Response$data2 === void 0 ? void 0 : (_s1Response$data2$dat = _s1Response$data2.data) === null || _s1Response$data2$dat === void 0 ? void 0 : _s1Response$data2$dat.parentTaskId) !== null && _s1Response$data$data3 !== void 0 ? _s1Response$data$data3 : ''
          };
        } catch (err) {
          error = err;
        }
      }
      reqIndexOptions.error = (_error5 = error) === null || _error5 === void 0 ? void 0 : _error5.message;
      if (!this.options.isAutomated && error) {
        throw error;
      }
    }
    const {
      actionDetails
    } = await this.handleResponseActionCreation(reqIndexOptions);
    return actionDetails;
  }
  async runscript(actionRequest, options) {
    if (!this.options.endpointService.experimentalFeatures.responseActionsSentinelOneRunScriptEnabled) {
      throw new _errors.ResponseActionsClientError(`'runscript' response action not supported for [${this.agentType}]. Feature disabled`);
    }
    const reqIndexOptions = {
      ...actionRequest,
      ...this.getMethodOptions(options),
      parameters: actionRequest.parameters,
      command: 'runscript'
    };
    if (!reqIndexOptions.error) {
      var _error6;
      let error = (await this.validateRequest(reqIndexOptions)).error;
      if (!error) {
        var _reqIndexOptions$para, _reqIndexOptions$para2;
        const scriptId = (_reqIndexOptions$para = (_reqIndexOptions$para2 = reqIndexOptions.parameters) === null || _reqIndexOptions$para2 === void 0 ? void 0 : _reqIndexOptions$para2.scriptId) !== null && _reqIndexOptions$para !== void 0 ? _reqIndexOptions$para : '';
        try {
          var _reqIndexOptions$para3, _s1Response$data$data4, _s1Response$data3, _s1Response$data3$dat;
          const s1Response = await this.sendAction(_constants.SUB_ACTION.EXECUTE_SCRIPT, {
            filter: {
              ids: actionRequest.endpoint_ids[0]
            },
            script: {
              scriptId,
              taskDescription: this.buildExternalComment(reqIndexOptions),
              requiresApproval: false,
              outputDestination: 'SentinelCloud',
              inputParams: (_reqIndexOptions$para3 = reqIndexOptions.parameters) === null || _reqIndexOptions$para3 === void 0 ? void 0 : _reqIndexOptions$para3.scriptInput
            }
          });
          reqIndexOptions.meta = {
            parentTaskId: (_s1Response$data$data4 = (_s1Response$data3 = s1Response.data) === null || _s1Response$data3 === void 0 ? void 0 : (_s1Response$data3$dat = _s1Response$data3.data) === null || _s1Response$data3$dat === void 0 ? void 0 : _s1Response$data3$dat.parentTaskId) !== null && _s1Response$data$data4 !== void 0 ? _s1Response$data$data4 : ''
          };
        } catch (err) {
          error = err;
        }
      }
      reqIndexOptions.error = (_error6 = error) === null || _error6 === void 0 ? void 0 : _error6.message;
      if (!this.options.isAutomated && error) {
        throw error;
      }
    }
    const {
      actionDetails
    } = await this.handleResponseActionCreation(reqIndexOptions);
    return actionDetails;
  }
  async getCustomScripts({
    osType
  } = {}) {
    var _scriptSearchResults$;
    if (!this.options.endpointService.experimentalFeatures.responseActionsSentinelOneRunScriptEnabled) {
      throw new _errors.ResponseActionsClientError(`'runscript' response action not supported for [${this.agentType}]`);
    }
    this.log.debug(`Retrieving list of scripts`);
    const s1ScriptQueryOptions = {
      sortBy: 'scriptName',
      sortOrder: 'asc',
      limit: 1000
    };
    if (osType) {
      s1ScriptQueryOptions.osTypes = osType;
    }
    const {
      data: scriptSearchResults
    } = await this.sendAction(_constants.SUB_ACTION.GET_REMOTE_SCRIPTS, s1ScriptQueryOptions);
    return {
      data: ((_scriptSearchResults$ = scriptSearchResults === null || scriptSearchResults === void 0 ? void 0 : scriptSearchResults.data) !== null && _scriptSearchResults$ !== void 0 ? _scriptSearchResults$ : []).map(({
        id,
        scriptName,
        scriptDescription,
        inputInstructions,
        inputExample,
        inputRequired,
        osTypes,
        shortFileName
      }) => {
        return {
          id,
          name: scriptName,
          description: `${scriptDescription !== null && scriptDescription !== void 0 ? scriptDescription : ''} ${inputInstructions ? `Input instructions: ${inputInstructions}` : ''}`.trim(),
          meta: {
            id,
            scriptDescription,
            osTypes,
            inputInstructions,
            inputExample,
            inputRequired,
            shortFileName
          }
        };
      })
    };
  }
  async processPendingActions({
    abortSignal,
    addToQueue
  }) {
    if (abortSignal.aborted) {
      return;
    }
    const addResponsesToQueueIfAny = responseList => {
      if (responseList.length > 0) {
        addToQueue(...responseList);
        this.sendActionResponseTelemetry(responseList);
      }
    };
    for await (const pendingActions of this.fetchAllPendingActions()) {
      if (abortSignal.aborted) {
        return;
      }
      const pendingActionsByType = (0, _lodash.groupBy)(pendingActions, 'action.EndpointActions.data.command');
      for (const [actionType, typePendingActions] of Object.entries(pendingActionsByType)) {
        if (abortSignal.aborted) {
          return;
        }
        switch (actionType) {
          case 'isolate':
          case 'unisolate':
            addResponsesToQueueIfAny(await this.checkPendingIsolateOrReleaseActions(typePendingActions, actionType));
            break;
          case 'running-processes':
            addResponsesToQueueIfAny(await this.checkPendingRunningProcessesAction(typePendingActions));
            break;
          case 'get-file':
            addResponsesToQueueIfAny(await this.checkPendingGetFileActions(typePendingActions));
            break;
          case 'kill-process':
            addResponsesToQueueIfAny(await this.checkPendingKillProcessActions(typePendingActions));
            break;
          case 'runscript':
            addResponsesToQueueIfAny(await this.checkPendingRunScriptActions(typePendingActions));
            break;
        }
      }
    }
  }

  /**
   * retrieve script info. for scripts that are used to handle Elastic response actions
   * @param scriptType
   * @param osType
   * @internal
   */
  async fetchScriptInfo(scriptType, osType) {
    var _scriptSearchResults$2;
    const searchQueryParams = {
      query: '',
      osTypes: osType
    };
    let buildScriptArgs = NOOP_THROW;
    let isDesiredScript = () => false;

    // Set the query value for filtering the list of scripts in S1
    switch (scriptType) {
      case 'kill-process':
        searchQueryParams.query = 'terminate';
        searchQueryParams.scriptType = 'action';
        isDesiredScript = scriptInfo => {
          var _scriptInfo$inputInst, _scriptInfo$inputInst2;
          return scriptInfo.creator === 'SentinelOne' && scriptInfo.creatorId === '-1' &&
          // Using single `-` (instead of double `--`) in match below to ensure both windows and macos/linux are matched
          /-terminate/i.test((_scriptInfo$inputInst = scriptInfo.inputInstructions) !== null && _scriptInfo$inputInst !== void 0 ? _scriptInfo$inputInst : '') && /-processes/i.test((_scriptInfo$inputInst2 = scriptInfo.inputInstructions) !== null && _scriptInfo$inputInst2 !== void 0 ? _scriptInfo$inputInst2 : '');
        };
        break;
      case 'running-processes':
        if (osType === 'windows') {
          throw new _errors.ResponseActionsClientError(`Retrieval of running processes for Windows host is not supported by SentinelOne`, 405);
        }
        searchQueryParams.query = 'process list';
        searchQueryParams.scriptType = 'dataCollection';
        isDesiredScript = scriptInfo => {
          return scriptInfo.creator === 'SentinelOne' && scriptInfo.creatorId === '-1';
        };
        break;
      default:
        throw new _errors.ResponseActionsClientError(`Unable to fetch SentinelOne script for OS [${osType}]. Unknown script type [${scriptType}]`);
    }
    const {
      data: scriptSearchResults
    } = await this.sendAction(_constants.SUB_ACTION.GET_REMOTE_SCRIPTS, searchQueryParams);
    const s1Script = ((_scriptSearchResults$2 = scriptSearchResults === null || scriptSearchResults === void 0 ? void 0 : scriptSearchResults.data) !== null && _scriptSearchResults$2 !== void 0 ? _scriptSearchResults$2 : []).find(isDesiredScript);
    if (!s1Script) {
      throw new _errors.ResponseActionsClientError(`Unable to find a script from SentinelOne to handle [${scriptType}] response action for host running [${osType}])`);
    }

    // Define the `buildScriptArgs` callback for the Script type
    switch (scriptType) {
      case 'kill-process':
        buildScriptArgs = args => {
          if (!args.processName) {
            throw new _errors.ResponseActionsClientError(`'processName' missing while building script args for [${s1Script.scriptName} (id: ${s1Script.id})] script`);
          }
          if (osType === 'windows') {
            return `-Terminate -Processes "${args.processName}" -Force`;
          }

          // Linux + Macos
          return `--terminate --processes "${args.processName}" --force`;
        };
        break;
      case 'running-processes':
        buildScriptArgs = () => '';
        break;
    }
    return {
      scriptId: s1Script.id,
      scriptInfo: s1Script,
      buildScriptArgs
    };
  }
  async fetchEsResponseDocForAgentId(actionId, agentId) {
    const agentResponse = (await this.fetchActionResponseEsDocs(actionId, [agentId]))[agentId];
    if (!agentResponse) {
      throw new _errors.ResponseActionAgentResponseEsDocNotFound(`Action ID [${actionId}] for agent ID [${agentId}] is still pending`, 404);
    }
    return agentResponse;
  }

  /**
   * Checks if the provided Isolate or Unisolate actions are complete and if so, then it builds the Response
   * document for them and returns it. (NOTE: the response is NOT written to ES - only returned)
   * @param actionRequests
   * @param command
   * @internal
   */
  async checkPendingIsolateOrReleaseActions(actionRequests, command) {
    const completedResponses = [];
    const actionsByAgentId = {};
    const warnings = [];

    // Create the `OR` clause that filters for each agent id and an updated date of greater than the date when
    // the isolate request was created
    const agentListQuery = actionRequests.reduce((acc, pendingActionData) => {
      var _action$meta;
      const action = pendingActionData.action;
      const s1AgentId = (_action$meta = action.meta) === null || _action$meta === void 0 ? void 0 : _action$meta.agentId;
      if (s1AgentId) {
        if (!actionsByAgentId[s1AgentId]) {
          actionsByAgentId[s1AgentId] = [];
        }
        actionsByAgentId[s1AgentId].push(action);
        acc.push({
          bool: {
            filter: [{
              term: {
                'sentinel_one.activity.agent.id': s1AgentId
              }
            }, {
              range: {
                'sentinel_one.activity.updated_at': {
                  gt: action['@timestamp']
                }
              }
            }]
          }
        });
      } else {
        // This is an edge case and should never happen. But just in case :-)
        warnings.push(`${command} response action ID [${action.EndpointActions.action_id}] missing SentinelOne agent ID, thus unable to check on it's status. Forcing it to complete as failure.`);
        completedResponses.push(this.buildActionResponseEsDoc({
          actionId: action.EndpointActions.action_id,
          agentId: Array.isArray(action.agent.id) ? action.agent.id[0] : action.agent.id,
          data: {
            command
          },
          error: {
            message: `Unable to very if action completed. SentinelOne agent id ('meta.agentId') missing on action request document!`
          }
        }));
      }
      return acc;
    }, []);
    if (agentListQuery.length > 0) {
      const query = {
        bool: {
          must: [{
            terms: {
              // Activity Types can be retrieved from S1 via API: `/web/api/v2.1/activities/types`
              'sentinel_one.activity.type': command === 'isolate' ? [
              // {
              //    "id": 1001
              //    "action": "Agent Disconnected From Network",
              //    "descriptionTemplate": "Agent {{ computer_name }} was disconnected from network.",
              // },
              1001,
              // {
              //    "id": 2010
              //    "action": "Agent Mitigation Report Quarantine Network Failed",
              //    "descriptionTemplate": "Agent {{ computer_name }} was unable to disconnect from network.",
              // },
              2010] : [
              // {
              //    "id": 1002
              //    "action": "Agent Reconnected To Network",
              //    "descriptionTemplate": "Agent {{ computer_name }} was connected to network.",
              // },
              1002]
            }
          }],
          should: agentListQuery,
          minimum_should_match: 1
        }
      };
      const searchRequestOptions = {
        index: _common.SENTINEL_ONE_ACTIVITY_INDEX_PATTERN,
        query,
        // There may be many documents for each host/agent, so we collapse it and only get back the
        // first one that came in after the isolate request was sent
        collapse: {
          field: 'sentinel_one.activity.agent.id',
          inner_hits: {
            name: 'first_found',
            size: 1,
            sort: [{
              'sentinel_one.activity.updated_at': 'asc'
            }]
          }
        },
        // we don't need the source. The document will be stored in `inner_hits.first_found`
        // due to use of `collapse
        _source: false,
        sort: [{
          'sentinel_one.activity.updated_at': {
            order: 'asc'
          }
        }],
        size: _constants2.ACTIONS_SEARCH_PAGE_SIZE
      };
      this.log.debug(() => `searching for ${command} responses from [${_common.SENTINEL_ONE_ACTIVITY_INDEX_PATTERN}] index with:\n${(0, _stringify.stringify)(searchRequestOptions, 15)}`);
      const searchResults = await this.options.esClient.search(searchRequestOptions).catch(_utils.catchAndWrapError);
      this.log.debug(() => `Search results for SentinelOne ${command} activity documents:\n${(0, _stringify.stringify)(searchResults)}`);
      for (const searchResultHit of searchResults.hits.hits) {
        var _searchResultHit$inne;
        const isolateActivityResponseDoc = (_searchResultHit$inne = searchResultHit.inner_hits) === null || _searchResultHit$inne === void 0 ? void 0 : _searchResultHit$inne.first_found.hits.hits[0];
        if (isolateActivityResponseDoc && isolateActivityResponseDoc._source) {
          const s1ActivityData = isolateActivityResponseDoc._source.sentinel_one.activity;

          // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
          const elasticDocId = isolateActivityResponseDoc._id;
          const s1AgentId = s1ActivityData.agent.id;
          const activityLogEntryId = s1ActivityData.id;
          const activityLogEntryType = s1ActivityData.type;
          const activityLogEntryDescription = s1ActivityData.description.primary;
          for (const actionRequest of actionsByAgentId[s1AgentId]) {
            completedResponses.push(this.buildActionResponseEsDoc({
              actionId: actionRequest.EndpointActions.action_id,
              agentId: Array.isArray(actionRequest.agent.id) ? actionRequest.agent.id[0] : actionRequest.agent.id,
              data: {
                command
              },
              error: activityLogEntryType === 2010 && command === 'isolate' ? {
                message: activityLogEntryDescription !== null && activityLogEntryDescription !== void 0 ? activityLogEntryDescription : `Action failed. SentinelOne activity log entry [${activityLogEntryId}] has a 'type' value of 2010 indicating a failure to disconnect`
              } : undefined,
              meta: {
                elasticDocId,
                activityLogEntryId,
                activityLogEntryType,
                activityLogEntryDescription
              }
            }));
          }
        }
      }
    } else {
      this.log.debug(`Nothing to search for. List of agents IDs is empty.`);
    }
    this.log.debug(() => `${completedResponses.length} ${command} action responses generated:\n${(0, _stringify.stringify)(completedResponses)}`);
    if (warnings.length > 0) {
      this.log.warn(warnings.join('\n'));
    }
    return completedResponses;
  }
  async checkPendingRunningProcessesAction(actionRequests) {
    const warnings = [];
    const completedResponses = [];
    for (const pendingAction of actionRequests) {
      var _actionRequest$meta2;
      const actionRequest = pendingAction.action;
      const s1ParentTaskId = (_actionRequest$meta2 = actionRequest.meta) === null || _actionRequest$meta2 === void 0 ? void 0 : _actionRequest$meta2.parentTaskId;
      if (!s1ParentTaskId) {
        completedResponses.push(this.buildActionResponseEsDoc({
          actionId: actionRequest.EndpointActions.action_id,
          agentId: Array.isArray(actionRequest.agent.id) ? actionRequest.agent.id[0] : actionRequest.agent.id,
          data: {
            command: 'running-processes',
            comment: ''
          },
          error: {
            message: `Action request missing SentinelOne 'parentTaskId' value - unable check on its status`
          }
        }));
        warnings.push(`Response Action [${actionRequest.EndpointActions.action_id}] is missing [meta.parentTaskId]! (should not have happened)`);
      } else {
        var _s1TaskStatusApiRespo;
        const s1TaskStatusApiResponse = await this.sendAction(_constants.SUB_ACTION.GET_REMOTE_SCRIPT_STATUS, {
          parentTaskId: s1ParentTaskId
        });
        if ((_s1TaskStatusApiRespo = s1TaskStatusApiResponse.data) !== null && _s1TaskStatusApiRespo !== void 0 && _s1TaskStatusApiRespo.data.length) {
          const processListScriptExecStatus = s1TaskStatusApiResponse.data.data[0];
          const taskState = this.calculateTaskState(processListScriptExecStatus);
          if (!taskState.isPending) {
            this.log.debug(`Action is completed - generating response doc for it`);
            const error = taskState.isError ? {
              message: `Action failed to execute in SentinelOne. message: ${taskState.message}`
            } : undefined;
            completedResponses.push(this.buildActionResponseEsDoc({
              actionId: actionRequest.EndpointActions.action_id,
              agentId: Array.isArray(actionRequest.agent.id) ? actionRequest.agent.id[0] : actionRequest.agent.id,
              data: {
                command: 'running-processes',
                comment: taskState.message,
                output: {
                  type: 'json',
                  content: {
                    code: '',
                    entries: []
                  }
                }
              },
              error,
              meta: {
                taskId: processListScriptExecStatus.id
              }
            }));
          }
        }
      }
    }
    this.log.debug(() => `${completedResponses.length} running-processes action responses generated:\n${(0, _stringify.stringify)(completedResponses)}`);
    if (warnings.length > 0) {
      this.log.warn(warnings.join('\n'));
    }
    return completedResponses;
  }
  async checkPendingGetFileActions(actionRequests) {
    const warnings = [];
    const completedResponses = [];
    const actionsByAgentAndBatchId = {};
    // Utility to create the key to lookup items in the `actionByAgentAndBatchId` grouping above
    const getLookupKey = (agentId, commandBatchUuid) => `${agentId}:${commandBatchUuid}`;
    const searchRequestOptions = {
      index: _common.SENTINEL_ONE_ACTIVITY_INDEX_PATTERN,
      size: _constants2.ACTIONS_SEARCH_PAGE_SIZE,
      query: {
        bool: {
          must: [{
            term: {
              // Activity Types can be retrieved from S1 via API: `/web/api/v2.1/activities/types`
              // {
              //   "action": "Agent Uploaded Fetched Files",
              //   "descriptionTemplate": "Agent {{ computer_name }} ({{ external_ip }}) successfully uploaded {{ filename }}.",
              //   "id": 80
              // },
              'sentinel_one.activity.type': 80
            }
          }],
          should: actionRequests.reduce((acc, {
            action
          }) => {
            var _action$meta2, _action$meta3;
            const s1AgentId = (_action$meta2 = action.meta) === null || _action$meta2 === void 0 ? void 0 : _action$meta2.agentId;
            const s1CommandBatchUUID = (_action$meta3 = action.meta) === null || _action$meta3 === void 0 ? void 0 : _action$meta3.commandBatchUuid;
            if (s1AgentId && s1CommandBatchUUID) {
              actionsByAgentAndBatchId[getLookupKey(s1AgentId, s1CommandBatchUUID)] = action;
              acc.push({
                bool: {
                  filter: [{
                    term: {
                      'sentinel_one.activity.agent.id': s1AgentId
                    }
                  }, {
                    term: {
                      'sentinel_one.activity.data.flattened.commandBatchUuid': s1CommandBatchUUID
                    }
                  }]
                }
              });
            } else {
              // This is an edge case and should never happen. But just in case :-)
              warnings.push(`get-file response action ID [${action.EndpointActions.action_id}] missing SentinelOne agent ID or commandBatchUuid value(s). Unable to check on it's status - forcing it to complete as a failure.`);
              completedResponses.push(this.buildActionResponseEsDoc({
                actionId: action.EndpointActions.action_id,
                agentId: Array.isArray(action.agent.id) ? action.agent.id[0] : action.agent.id,
                data: {
                  command: 'get-file'
                },
                error: {
                  message: `Unable to very if action completed. SentinelOne agent id or commandBatchUuid missing on action request document!`
                }
              }));
            }
            return acc;
          }, []),
          minimum_should_match: 1
        }
      }
    };
    if (Object.keys(actionsByAgentAndBatchId).length) {
      this.log.debug(() => `searching for get-file responses from [${_common.SENTINEL_ONE_ACTIVITY_INDEX_PATTERN}] index with:\n${(0, _stringify.stringify)(searchRequestOptions, 15)}`);
      const searchResults = await this.options.esClient.search(searchRequestOptions).catch(_utils.catchAndWrapError);
      this.log.debug(() => `Search results for SentinelOne get-file activity documents:\n${(0, _stringify.stringify)(searchResults)}`);
      for (const s1Hit of searchResults.hits.hits) {
        var _s1ActivityDoc$sentin, _s1ActivityDoc$sentin2;
        const s1ActivityDoc = s1Hit._source;
        const s1AgentId = s1ActivityDoc === null || s1ActivityDoc === void 0 ? void 0 : s1ActivityDoc.sentinel_one.activity.agent.id;
        const s1CommandBatchUuid = (_s1ActivityDoc$sentin = s1ActivityDoc === null || s1ActivityDoc === void 0 ? void 0 : s1ActivityDoc.sentinel_one.activity.data.flattened.commandBatchUuid) !== null && _s1ActivityDoc$sentin !== void 0 ? _s1ActivityDoc$sentin : '';
        const activityLogEntryId = (_s1ActivityDoc$sentin2 = s1ActivityDoc === null || s1ActivityDoc === void 0 ? void 0 : s1ActivityDoc.sentinel_one.activity.id) !== null && _s1ActivityDoc$sentin2 !== void 0 ? _s1ActivityDoc$sentin2 : '';
        if (s1AgentId && s1CommandBatchUuid) {
          const actionRequest = actionsByAgentAndBatchId[getLookupKey(s1AgentId, s1CommandBatchUuid)];
          if (actionRequest) {
            var _s1ActivityDoc$sentin3, _s1ActivityDoc$sentin4, _s1ActivityDoc$sentin5, _s1ActivityDoc$sentin6;
            const downloadUrl = (_s1ActivityDoc$sentin3 = s1ActivityDoc === null || s1ActivityDoc === void 0 ? void 0 : s1ActivityDoc.sentinel_one.activity.data.downloaded.url) !== null && _s1ActivityDoc$sentin3 !== void 0 ? _s1ActivityDoc$sentin3 : '';
            const error = !downloadUrl ? {
              message: `File retrieval failed (No download URL defined in SentinelOne activity log id [${activityLogEntryId}])`
            } : undefined;
            completedResponses.push(this.buildActionResponseEsDoc({
              actionId: actionRequest.EndpointActions.action_id,
              agentId: Array.isArray(actionRequest.agent.id) ? actionRequest.agent.id[0] : actionRequest.agent.id,
              data: {
                command: 'get-file',
                comment: (_s1ActivityDoc$sentin4 = s1ActivityDoc === null || s1ActivityDoc === void 0 ? void 0 : s1ActivityDoc.sentinel_one.activity.description.primary) !== null && _s1ActivityDoc$sentin4 !== void 0 ? _s1ActivityDoc$sentin4 : '',
                output: {
                  type: 'json',
                  content: {
                    // code applies only to Endpoint agents
                    code: '',
                    // We don't know the file size for S1 retrieved files
                    zip_size: 0,
                    // We don't have the contents of the zip file for S1
                    contents: []
                  }
                }
              },
              error,
              meta: {
                activityLogEntryId,
                // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                elasticDocId: s1Hit._id,
                downloadUrl,
                createdAt: (_s1ActivityDoc$sentin5 = s1ActivityDoc === null || s1ActivityDoc === void 0 ? void 0 : s1ActivityDoc.sentinel_one.activity.updated_at) !== null && _s1ActivityDoc$sentin5 !== void 0 ? _s1ActivityDoc$sentin5 : '',
                filename: (_s1ActivityDoc$sentin6 = s1ActivityDoc === null || s1ActivityDoc === void 0 ? void 0 : s1ActivityDoc.sentinel_one.activity.data.flattened.filename) !== null && _s1ActivityDoc$sentin6 !== void 0 ? _s1ActivityDoc$sentin6 : ''
              }
            }));
          } else {
            warnings.push(`Activity log entry ${s1Hit._id} was a matched, but no action request for it (should not happen)`);
          }
        }
      }
    } else {
      this.log.debug(`Nothing to search for. No pending get-file actions`);
    }
    this.log.debug(() => `${completedResponses.length} get-file action responses generated:\n${(0, _stringify.stringify)(completedResponses)}`);
    if (warnings.length > 0) {
      this.log.warn(warnings.join('\n'));
    }
    return completedResponses;
  }

  /**
   * Calculates the state of a SentinelOne Task using the response from their task status API. It
   * returns a normalized object with basic info derived from the task status value
   * @param taskStatusRecord
   * @internal
   */
  calculateTaskState(taskStatusRecord) {
    var _ref, _taskStatusRecord$det;
    const taskStatusValue = taskStatusRecord.status;
    let message = (_ref = (_taskStatusRecord$det = taskStatusRecord.detailedStatus) !== null && _taskStatusRecord$det !== void 0 ? _taskStatusRecord$det : taskStatusRecord.statusDescription) !== null && _ref !== void 0 ? _ref : taskStatusValue;
    let isPending;
    let isError;
    switch (taskStatusValue) {
      // PENDING STATUSES ------------------------------------------
      case 'created':
      case 'pending':
      case 'pending_user_action':
      case 'scheduled':
      case 'in_progress':
      case 'partially_completed':
        isPending = true;
        isError = true;
        break;

      // COMPLETE STATUSES ------------------------------------------
      case 'canceled':
        isPending = false;
        isError = true;
        message = `SentinelOne Parent Task Id [${taskStatusRecord.parentTaskId}] was canceled${taskStatusRecord.detailedStatus ? ` - ${taskStatusRecord.detailedStatus}` : ''}`;
        break;
      case 'completed':
        isPending = false;
        isError = false;
        break;
      case 'expired':
        isPending = false;
        isError = true;
        break;
      case 'failed':
        isPending = false;
        isError = true;
        break;
      default:
        isPending = false;
        isError = true;
        message = `Unable to determine task state - unknown SentinelOne task status value [${taskStatusRecord}] for task parent id [${taskStatusRecord.parentTaskId}]`;
    }
    return {
      isPending,
      isError,
      message
    };
  }
  async checkPendingKillProcessActions(actionRequests) {
    const warnings = [];
    const completedResponses = [];
    for (const pendingAction of actionRequests) {
      var _actionRequest$meta3;
      const actionRequest = pendingAction.action;
      const s1ParentTaskId = (_actionRequest$meta3 = actionRequest.meta) === null || _actionRequest$meta3 === void 0 ? void 0 : _actionRequest$meta3.parentTaskId;
      if (!s1ParentTaskId) {
        completedResponses.push(this.buildActionResponseEsDoc({
          actionId: actionRequest.EndpointActions.action_id,
          agentId: Array.isArray(actionRequest.agent.id) ? actionRequest.agent.id[0] : actionRequest.agent.id,
          data: {
            command: 'kill-process',
            comment: ''
          },
          error: {
            message: `Action request missing SentinelOne 'parentTaskId' value - unable check on its status`
          }
        }));
        warnings.push(`Response Action [${actionRequest.EndpointActions.action_id}] is missing [meta.parentTaskId]! (should not have happened)`);
      } else {
        var _s1TaskStatusApiRespo2;
        const s1TaskStatusApiResponse = await this.sendAction(_constants.SUB_ACTION.GET_REMOTE_SCRIPT_STATUS, {
          parentTaskId: s1ParentTaskId
        });
        if ((_s1TaskStatusApiRespo2 = s1TaskStatusApiResponse.data) !== null && _s1TaskStatusApiRespo2 !== void 0 && _s1TaskStatusApiRespo2.data.length) {
          const killProcessStatus = s1TaskStatusApiResponse.data.data[0];
          const taskState = this.calculateTaskState(killProcessStatus);
          if (!taskState.isPending) {
            var _killProcessStatus$st, _actionRequest$Endpoi;
            this.log.debug(`Action is completed - generating response doc for it`);
            const error = taskState.isError ? {
              message: `Action failed to execute in SentinelOne. message: ${taskState.message}`
            } : undefined;
            completedResponses.push(this.buildActionResponseEsDoc({
              actionId: actionRequest.EndpointActions.action_id,
              agentId: Array.isArray(actionRequest.agent.id) ? actionRequest.agent.id[0] : actionRequest.agent.id,
              data: {
                command: 'kill-process',
                comment: taskState.message,
                output: {
                  type: 'json',
                  content: {
                    code: (_killProcessStatus$st = killProcessStatus.statusCode) !== null && _killProcessStatus$st !== void 0 ? _killProcessStatus$st : killProcessStatus.status,
                    command: actionRequest.EndpointActions.data.command,
                    process_name: (_actionRequest$Endpoi = actionRequest.EndpointActions.data.parameters) === null || _actionRequest$Endpoi === void 0 ? void 0 : _actionRequest$Endpoi.process_name
                  }
                }
              },
              error,
              meta: {
                taskId: killProcessStatus.id
              }
            }));
          }
        }
      }
    }
    this.log.debug(() => `${completedResponses.length} kill-process action responses generated:\n${(0, _stringify.stringify)(completedResponses)}`);
    if (warnings.length > 0) {
      this.log.warn(warnings.join('\n'));
    }
    return completedResponses;
  }
  async checkPendingRunScriptActions(actionRequests) {
    // TODO:PT refactor the "checkPending*()" methods to maybe centralize common logic

    const warnings = [];
    const completedResponses = [];
    for (const pendingAction of actionRequests) {
      var _actionRequest$meta4;
      const actionRequest = pendingAction.action;
      const s1ParentTaskId = (_actionRequest$meta4 = actionRequest.meta) === null || _actionRequest$meta4 === void 0 ? void 0 : _actionRequest$meta4.parentTaskId;
      if (!s1ParentTaskId) {
        completedResponses.push(this.buildActionResponseEsDoc({
          actionId: actionRequest.EndpointActions.action_id,
          agentId: Array.isArray(actionRequest.agent.id) ? actionRequest.agent.id[0] : actionRequest.agent.id,
          data: {
            command: 'runscript',
            comment: ''
          },
          error: {
            message: `Action request missing SentinelOne 'parentTaskId' value - unable check on its status`
          }
        }));
        warnings.push(`Response Action [${actionRequest.EndpointActions.action_id}] is missing [meta.parentTaskId]! (should not have happened)`);
      } else {
        var _s1TaskStatusApiRespo3;
        const s1TaskStatusApiResponse = await this.sendAction(_constants.SUB_ACTION.GET_REMOTE_SCRIPT_STATUS, {
          parentTaskId: s1ParentTaskId
        });
        if ((_s1TaskStatusApiRespo3 = s1TaskStatusApiResponse.data) !== null && _s1TaskStatusApiRespo3 !== void 0 && _s1TaskStatusApiRespo3.data.length) {
          const s1ScriptStatus = s1TaskStatusApiResponse.data.data[0];
          const taskState = this.calculateTaskState(s1ScriptStatus);
          if (!taskState.isPending) {
            var _s1ScriptStatus$statu;
            this.log.debug(`Action is completed - generating response doc for it`);
            const error = taskState.isError ? {
              message: `Action failed to execute in SentinelOne. Message: ${taskState.message}`
            } : undefined;
            completedResponses.push(this.buildActionResponseEsDoc({
              actionId: actionRequest.EndpointActions.action_id,
              agentId: Array.isArray(actionRequest.agent.id) ? actionRequest.agent.id[0] : actionRequest.agent.id,
              data: {
                command: 'runscript',
                comment: taskState.message,
                output: {
                  type: 'json',
                  content: {
                    stdout: '',
                    stderr: '',
                    code: (_s1ScriptStatus$statu = s1ScriptStatus.statusCode) !== null && _s1ScriptStatus$statu !== void 0 ? _s1ScriptStatus$statu : s1ScriptStatus.status
                  }
                }
              },
              error,
              meta: {
                taskId: s1ScriptStatus.id
              }
            }));
          }
        }
      }
    }
    this.log.debug(() => `${completedResponses.length} runscript action responses generated:\n${(0, _stringify.stringify)(completedResponses)}`);
    if (warnings.length > 0) {
      this.log.warn(warnings.join('\n'));
    }
    return completedResponses;
  }
}
exports.SentinelOneActionsClient = SentinelOneActionsClient;