"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerKibanaFunction = registerKibanaFunction;
var _axios = _interopRequireDefault(require("axios"));
var _url = require("url");
var _lodash = require("lodash");
var _common = require("@kbn/spaces-plugin/common");
var _ = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerKibanaFunction({
  functions,
  resources
}) {
  functions.registerFunction({
    name: _.KIBANA_FUNCTION_NAME,
    description: 'Call Kibana APIs on behalf of the user. Only call this function when the user has explicitly requested it, and you know how to call it, for example by querying the knowledge base or having the user explain it to you. Assume that pathnames, bodies and query parameters may have changed since your knowledge cut off date.',
    descriptionForUser: 'Call Kibana APIs on behalf of the user',
    parameters: {
      type: 'object',
      properties: {
        method: {
          type: 'string',
          description: 'The HTTP method of the Kibana endpoint',
          enum: ['GET', 'PUT', 'POST', 'DELETE', 'PATCH']
        },
        pathname: {
          type: 'string',
          description: 'The pathname of the Kibana endpoint, excluding query parameters'
        },
        query: {
          type: 'object',
          description: 'The query parameters, as an object'
        },
        body: {
          type: 'object',
          description: 'The body of the request'
        }
      },
      required: ['method', 'pathname']
    }
  }, async ({
    arguments: {
      method,
      pathname,
      body,
      query
    }
  }, signal) => {
    const {
      request,
      logger
    } = resources;
    const requestUrl = request.rewrittenUrl || request.url;
    const core = await resources.plugins.core.start();
    function getParsedPublicBaseUrl() {
      const {
        publicBaseUrl
      } = core.http.basePath;
      if (!publicBaseUrl) {
        const errorMessage = `Cannot invoke Kibana tool: "server.publicBaseUrl" must be configured in kibana.yml`;
        logger.error(errorMessage);
        throw new Error(errorMessage);
      }
      const parsedBaseUrl = new URL(publicBaseUrl);
      return parsedBaseUrl;
    }
    function getPathnameWithSpaceId() {
      const {
        serverBasePath
      } = core.http.basePath;
      const {
        spaceId
      } = (0, _common.getSpaceIdFromPath)(requestUrl.pathname, serverBasePath);
      const pathnameWithSpaceId = (0, _common.addSpaceIdToPath)(serverBasePath, spaceId, pathname);
      return pathnameWithSpaceId;
    }
    const parsedPublicBaseUrl = getParsedPublicBaseUrl();
    const nextUrl = {
      host: parsedPublicBaseUrl.host,
      protocol: parsedPublicBaseUrl.protocol,
      pathname: getPathnameWithSpaceId(),
      query: query ? query : undefined
    };
    logger.info(`Calling Kibana API by forwarding request from "${requestUrl}" to: "${method} ${(0, _url.format)(nextUrl)}"`);
    const copiedHeaderNames = ['accept-encoding', 'accept-language', 'accept', 'authorization', 'content-type', 'cookie', 'kbn-build-number', 'kbn-version', 'origin', 'referer', 'user-agent', 'x-elastic-internal-origin', 'x-elastic-product-origin', 'x-kbn-context'];
    const headers = (0, _lodash.pickBy)(request.headers, (value, key) => {
      return copiedHeaderNames.includes(key.toLowerCase()) || key.toLowerCase().startsWith('sec-');
    });
    try {
      const response = await (0, _axios.default)({
        method,
        headers,
        url: (0, _url.format)(nextUrl),
        data: body ? JSON.stringify(body) : undefined,
        signal
      });
      return {
        content: response.data
      };
    } catch (e) {
      logger.error(`Error calling Kibana API: ${method} ${(0, _url.format)(nextUrl)}. Failed with ${e}`);
      throw e;
    }
  });
}