"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.GlobalVisitorContext = void 0;
var contexts = _interopRequireWildcard(require("./contexts"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Global shared visitor context available to all visitors when visiting the AST.
 * It contains the shared data, which can be accessed and modified by all visitors.
 */
class GlobalVisitorContext {
  constructor(
  /**
   * Visitor methods, used internally by the visitor to traverse the AST.
   * @protected
   */
  methods,
  /**
   * Shared data, which can be accessed and modified by all visitors.
   */
  data) {
    this.methods = methods;
    this.data = data;
  }
  assertMethodExists(name) {
    if (!Array.isArray(name)) {
      name = [name];
    }
    for (const n of name) {
      if (this.methods[n]) return;
    }
    throw new Error(`${name}() method is not defined`);
  }
  visitWithSpecificContext(method, context, input) {
    this.assertMethodExists(method);
    return this.methods[method](context, input);
  }

  // #region Command visiting ----------------------------------------------------------

  visitCommandGeneric(parent, node, input) {
    this.assertMethodExists('visitCommand');
    const context = new contexts.CommandVisitorContext(this, node, parent);
    const output = this.methods.visitCommand(context, input);
    return output;
  }
  visitCommand(parent, commandNode, input) {
    switch (commandNode.name) {
      case 'from':
        {
          if (!this.methods.visitFromCommand) break;
          return this.visitFromCommand(parent, commandNode, input);
        }
      case 'limit':
        {
          if (!this.methods.visitLimitCommand) break;
          return this.visitLimitCommand(parent, commandNode, input);
        }
      case 'explain':
        {
          if (!this.methods.visitExplainCommand) break;
          return this.visitExplainCommand(parent, commandNode, input);
        }
      case 'row':
        {
          if (!this.methods.visitRowCommand) break;
          return this.visitRowCommand(parent, commandNode, input);
        }
      case 'ts':
        {
          if (!this.methods.visitTimeseriesCommand) break;
          return this.visitTimeseriesCommand(parent, commandNode, input);
        }
      case 'show':
        {
          if (!this.methods.visitShowCommand) break;
          return this.visitShowCommand(parent, commandNode, input);
        }
      case 'meta':
        {
          if (!this.methods.visitMetaCommand) break;
          return this.visitMetaCommand(parent, commandNode, input);
        }
      case 'eval':
        {
          if (!this.methods.visitEvalCommand) break;
          return this.visitEvalCommand(parent, commandNode, input);
        }
      case 'stats':
        {
          if (!this.methods.visitStatsCommand) break;
          return this.visitStatsCommand(parent, commandNode, input);
        }
      case 'inline stats':
        {
          if (!this.methods.visitInlineStatsCommand) break;
          return this.visitInlineStatsCommand(parent, commandNode, input);
        }
      case 'lookup':
        {
          if (!this.methods.visitLookupCommand) break;
          return this.visitLookupCommand(parent, commandNode, input);
        }
      case 'keep':
        {
          if (!this.methods.visitKeepCommand) break;
          return this.visitKeepCommand(parent, commandNode, input);
        }
      case 'sort':
        {
          if (!this.methods.visitSortCommand) break;
          return this.visitSortCommand(parent, commandNode, input);
        }
      case 'where':
        {
          if (!this.methods.visitWhereCommand) break;
          return this.visitWhereCommand(parent, commandNode, input);
        }
      case 'drop':
        {
          if (!this.methods.visitDropCommand) break;
          return this.visitDropCommand(parent, commandNode, input);
        }
      case 'rename':
        {
          if (!this.methods.visitRenameCommand) break;
          return this.visitRenameCommand(parent, commandNode, input);
        }
      case 'dissect':
        {
          if (!this.methods.visitDissectCommand) break;
          return this.visitDissectCommand(parent, commandNode, input);
        }
      case 'grok':
        {
          if (!this.methods.visitGrokCommand) break;
          return this.visitGrokCommand(parent, commandNode, input);
        }
      case 'enrich':
        {
          if (!this.methods.visitEnrichCommand) break;
          return this.visitEnrichCommand(parent, commandNode, input);
        }
      case 'mv_expand':
        {
          if (!this.methods.visitMvExpandCommand) break;
          return this.visitMvExpandCommand(parent, commandNode, input);
        }
      case 'join':
        {
          if (!this.methods.visitJoinCommand) break;
          return this.visitJoinCommand(parent, commandNode, input);
        }
      case 'rerank':
        {
          if (!this.methods.visitRerankCommand) break;
          return this.visitRerankCommand(parent, commandNode, input);
        }
      case 'change_point':
        {
          if (!this.methods.visitChangePointCommand) break;
          return this.visitChangePointCommand(parent, commandNode, input);
        }
      case 'fork':
        {
          if (!this.methods.visitForkCommand) break;
          return this.visitForkCommand(parent, commandNode, input);
        }
      case 'completion':
        {
          if (!this.methods.visitCompletionCommand) break;
          return this.visitCompletionCommand(parent, commandNode, input);
        }
      case 'sample':
        {
          if (!this.methods.visitSampleCommand) break;
          return this.visitSampleCommand(parent, commandNode, input);
        }
      case 'fuse':
        {
          if (!this.methods.visitFuseCommand) break;
          return this.visitFuseCommand(parent, commandNode, input);
        }
    }
    return this.visitCommandGeneric(parent, commandNode, input);
  }
  visitFromCommand(parent, node, input) {
    const context = new contexts.FromCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitFromCommand', context, input);
  }
  visitLimitCommand(parent, node, input) {
    const context = new contexts.LimitCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitLimitCommand', context, input);
  }
  visitExplainCommand(parent, node, input) {
    const context = new contexts.ExplainCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitExplainCommand', context, input);
  }
  visitRowCommand(parent, node, input) {
    const context = new contexts.RowCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitRowCommand', context, input);
  }
  visitTimeseriesCommand(parent, node, input) {
    const context = new contexts.TimeseriesCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitTimeseriesCommand', context, input);
  }
  visitShowCommand(parent, node, input) {
    const context = new contexts.ShowCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitShowCommand', context, input);
  }
  visitMetaCommand(parent, node, input) {
    const context = new contexts.MetaCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitMetaCommand', context, input);
  }
  visitEvalCommand(parent, node, input) {
    const context = new contexts.EvalCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitEvalCommand', context, input);
  }
  visitStatsCommand(parent, node, input) {
    const context = new contexts.StatsCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitStatsCommand', context, input);
  }
  visitInlineStatsCommand(parent, node, input) {
    const context = new contexts.InlineStatsCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitInlineStatsCommand', context, input);
  }
  visitLookupCommand(parent, node, input) {
    const context = new contexts.LookupCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitLookupCommand', context, input);
  }
  visitKeepCommand(parent, node, input) {
    const context = new contexts.KeepCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitKeepCommand', context, input);
  }
  visitSortCommand(parent, node, input) {
    const context = new contexts.SortCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitSortCommand', context, input);
  }
  visitWhereCommand(parent, node, input) {
    const context = new contexts.WhereCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitWhereCommand', context, input);
  }
  visitDropCommand(parent, node, input) {
    const context = new contexts.DropCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitDropCommand', context, input);
  }
  visitRenameCommand(parent, node, input) {
    const context = new contexts.RenameCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitRenameCommand', context, input);
  }
  visitDissectCommand(parent, node, input) {
    const context = new contexts.DissectCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitDissectCommand', context, input);
  }
  visitGrokCommand(parent, node, input) {
    const context = new contexts.GrokCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitGrokCommand', context, input);
  }
  visitEnrichCommand(parent, node, input) {
    const context = new contexts.EnrichCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitEnrichCommand', context, input);
  }
  visitMvExpandCommand(parent, node, input) {
    const context = new contexts.MvExpandCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitMvExpandCommand', context, input);
  }
  visitJoinCommand(parent, node, input) {
    const context = new contexts.JoinCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitJoinCommand', context, input);
  }
  visitRerankCommand(parent, node, input) {
    const context = new contexts.RerankCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitRerankCommand', context, input);
  }
  visitChangePointCommand(parent, node, input) {
    const context = new contexts.ChangePointCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitChangePointCommand', context, input);
  }
  visitForkCommand(parent, node, input) {
    const context = new contexts.ForkCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitForkCommand', context, input);
  }
  visitCompletionCommand(parent, node, input) {
    const context = new contexts.CompletionCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitCompletionCommand', context, input);
  }
  visitSampleCommand(parent, node, input) {
    const context = new contexts.ForkCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitSampleCommand', context, input);
  }
  visitFuseCommand(parent, node, input) {
    const context = new contexts.FuseCommandVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitFuseCommand', context, input);
  }

  // #endregion

  // #region Expression visiting -------------------------------------------------------

  visitExpressionGeneric(parent, node, input) {
    this.assertMethodExists('visitExpression');
    const context = new contexts.ExpressionVisitorContext(this, node, parent);
    const output = this.methods.visitExpression(context, input);
    return output;
  }
  visitExpression(parent, expressionNode, input) {
    if (Array.isArray(expressionNode)) {
      throw new Error('should not happen');
    }
    switch (expressionNode.type) {
      case 'column':
        {
          if (!this.methods.visitColumnExpression) break;
          return this.visitColumnExpression(parent, expressionNode, input);
        }
      case 'source':
        {
          if (!this.methods.visitSourceExpression) break;
          return this.visitSourceExpression(parent, expressionNode, input);
        }
      case 'function':
        {
          if (!this.methods.visitFunctionCallExpression) break;
          return this.visitFunctionCallExpression(parent, expressionNode, input);
        }
      case 'literal':
        {
          if (!this.methods.visitLiteralExpression) break;
          return this.visitLiteralExpression(parent, expressionNode, input);
        }
      case 'list':
        {
          if (!this.methods.visitListLiteralExpression) break;
          return this.visitListLiteralExpression(parent, expressionNode, input);
        }
      case 'inlineCast':
        {
          if (!this.methods.visitInlineCastExpression) break;
          return this.visitInlineCastExpression(parent, expressionNode, input);
        }
      case 'order':
        {
          if (!this.methods.visitOrderExpression) break;
          return this.visitOrderExpression(parent, expressionNode, input);
        }
      case 'identifier':
        {
          if (!this.methods.visitIdentifierExpression) break;
          return this.visitIdentifierExpression(parent, expressionNode, input);
        }
      case 'map':
        {
          if (!this.methods.visitMapExpression) break;
          return this.visitMapExpression(parent, expressionNode, input);
        }
      case 'map-entry':
        {
          if (!this.methods.visitMapEntryExpression) break;
          return this.visitMapEntryExpression(parent, expressionNode, input);
        }
      case 'query':
        {
          if (!this.methods.visitQuery || expressionNode.type !== 'query') break;
          return this.visitQuery(parent, expressionNode, input);
        }
    }
    return this.visitExpressionGeneric(parent, expressionNode, input);
  }
  visitQuery(parent, node, input) {
    const context = new contexts.QueryVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitQuery', context, input);
  }
  visitColumnExpression(parent, node, input) {
    const context = new contexts.ColumnExpressionVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitColumnExpression', context, input);
  }
  visitSourceExpression(parent, node, input) {
    const context = new contexts.SourceExpressionVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitSourceExpression', context, input);
  }
  visitFunctionCallExpression(parent, node, input) {
    const context = new contexts.FunctionCallExpressionVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitFunctionCallExpression', context, input);
  }
  visitLiteralExpression(parent, node, input) {
    const context = new contexts.LiteralExpressionVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitLiteralExpression', context, input);
  }
  visitListLiteralExpression(parent, node, input) {
    const context = new contexts.ListLiteralExpressionVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitListLiteralExpression', context, input);
  }
  visitInlineCastExpression(parent, node, input) {
    const context = new contexts.InlineCastExpressionVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitInlineCastExpression', context, input);
  }
  visitOrderExpression(parent, node, input) {
    const context = new contexts.OrderExpressionVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitOrderExpression', context, input);
  }
  visitIdentifierExpression(parent, node, input) {
    const context = new contexts.IdentifierExpressionVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitIdentifierExpression', context, input);
  }
  visitMapExpression(parent, node, input) {
    const context = new contexts.MapExpressionVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitMapExpression', context, input);
  }
  visitMapEntryExpression(parent, node, input) {
    const context = new contexts.MapEntryExpressionVisitorContext(this, node, parent);
    return this.visitWithSpecificContext('visitMapEntryExpression', context, input);
  }
}

// #endregion
exports.GlobalVisitorContext = GlobalVisitorContext;