"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.summarizeCommand = exports.summarize = exports.list = exports.byIndex = void 0;
var _is = require("../../../ast/is");
var _walker = require("../../../walker");
var generic = _interopRequireWildcard(require("../../generic"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Lists all "JOIN" commands in the query AST.
 *
 * @param ast The root AST node to search for "JOIN" commands.
 * @returns A collection of "JOIN" commands.
 */
const list = ast => {
  return generic.commands.list(ast, cmd => cmd.name === 'join');
};

/**
 * Retrieves the "JOIN" command at the specified index in order of appearance.
 *
 * @param ast The root AST node to search for "JOIN" commands.
 * @param index The index of the "JOIN" command to retrieve.
 * @returns The "JOIN" command at the specified index, if any.
 */
exports.list = list;
const byIndex = (ast, index) => {
  return [...list(ast)][index];
};
exports.byIndex = byIndex;
const getSource = node => _walker.Walker.match(node, {
  type: 'source'
});
const getIdentifier = node => _walker.Walker.match(node, {
  type: 'identifier'
});

/**
 * Summarizes a single JOIN command.
 *
 * @param command JOIN command to summarize.
 * @returns Returns a summary of the JOIN command.
 */
const summarizeCommand = command => {
  const firstArg = command.args[0];
  let index;
  let alias;
  const conditions = [];
  if ((0, _is.isAsExpression)(firstArg)) {
    index = getSource(firstArg.args[0]);
    alias = getIdentifier(firstArg.args[1]);
  } else {
    index = getSource(firstArg);
  }
  const on = generic.commands.options.find(command, ({
    name
  }) => name === 'on');
  conditions.push(...((on === null || on === void 0 ? void 0 : on.args) || []));
  const target = {
    index: index,
    alias
  };
  const summary = {
    target,
    conditions
  };
  return summary;
};

/**
 * Summarizes all JOIN commands in the query.
 *
 * @param query Query to summarize.
 * @returns Returns a list of summaries for all JOIN commands in the query in
 *     order of appearance.
 */
exports.summarizeCommand = summarizeCommand;
const summarize = query => {
  const summaries = [];
  for (const command of list(query)) {
    const summary = summarizeCommand(command);
    summaries.push(summary);
  }
  return summaries;
};
exports.summarize = summarize;